<?php
// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set Arabic language headers
header('Content-Type: text/html; charset=utf-8');

// Include database connection
require_once 'db.php';

// API endpoint (replace with your actual API URL)
$apiUrl = 'http://62.164.219.146:8000/api/v2/image/validate';
$apiKey = '9tZwimURlQMdn8VaI2IAGSFQ4fgbJVd9'; // Replace with your actual API key

// Manual CAPTCHA configuration
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['captcha'])) {
    $_SESSION['captcha'] = generateCaptcha();
}

// Define steps
$step = 1; // Default step
$apiResponse = null;
$error = null;
$success = null;
$imagePreview = null;
$studentNumber = '';
$studentName = '';
$paymentVerified = false;
$existingImage = null;
$overwriteImage = false;
$validationResults = [];
$allValid = false;
$showValidationResults = false;
$captchaError = false;

// Generate CAPTCHA function
function generateCaptcha() {
    $length = 6;
    $characters = '23456789ABCDEFGHJKLMNPQRSTUVWXYZ'; // Removed similar looking characters
    $captcha = '';
    for ($i = 0; $i < $length; $i++) {
        $captcha .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $captcha;
}

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['step']) && $_POST['step'] == '1') {
        // Step 1: Check student payment
        $step = 1;
        $studentNumber = trim($_POST['student_number'] ?? '');
        
        if (empty($studentNumber)) {
            $error = "الرجاء إدخال رقم الطالب.";
        } else {
            // Check if student has paid
            $stmt = $mysqli1->prepare("SELECT student_name FROM payments WHERE student_id = ? LIMIT 1");
            if ($stmt) {
                $stmt->bind_param("s", $studentNumber);
                $stmt->execute();
                $stmt->store_result();
                
                if ($stmt->num_rows > 0) {
                    $stmt->bind_result($studentName);
                    $stmt->fetch();
                    $paymentVerified = true;
                    
                    // Check if image already exists
                    $imageDir = 'uploads/';
                    $imagePattern = $imageDir . $studentNumber . '.*';
                    $existingFiles = glob($imagePattern);
                    
                    if (!empty($existingFiles)) {
                        $existingImage = $existingFiles[0];
                    }
                    
                    // Move to step 2 if payment is verified
                    $step = 2;
                } else {
                    $error = "❌ لم يتم العثور على دفعة لهذا الرقم. الرجاء التحقق من رقم الطالب أو دفع الرسوم أولاً.";
                    $paymentVerified = false;
                }
                $stmt->close();
            } else {
                $error = "خطأ في التحقق من قاعدة البيانات.";
                $paymentVerified = false;
            }
        }
    } elseif (isset($_POST['step']) && $_POST['step'] == '2') {
        // Step 2: Upload and validate image
        $step = 2;
        $studentNumber = trim($_POST['student_number'] ?? '');
        $overwriteImage = isset($_POST['overwrite_image']) && $_POST['overwrite_image'] == '1';
        
        // Validate student number again
        if (empty($studentNumber)) {
            $error = "رقم الطالب غير صالح. الرجاء العودة للخطوة الأولى.";
            $step = 1;
        } else {
            // Check payment again for security
            $stmt = $mysqli1->prepare("SELECT student_name FROM payments WHERE student_id = ? LIMIT 1");
            if ($stmt) {
                $stmt->bind_param("s", $studentNumber);
                $stmt->execute();
                $stmt->store_result();
                
                if ($stmt->num_rows > 0) {
                    $stmt->bind_result($studentName);
                    $stmt->fetch();
                    $paymentVerified = true;
                    
                    // Check if image already exists
                    $imageDir = 'uploads/';
                    $imagePattern = $imageDir . $studentNumber . '.*';
                    $existingFiles = glob($imagePattern);
                    
                    if (!empty($existingFiles)) {
                        $existingImage = $existingFiles[0];
                        
                        // If overwrite is not confirmed and image exists, show warning
                        if (!$overwriteImage && $existingImage) {
                            $step = 2; // Stay on step 2 but show overwrite warning
                        } else {
                            // Proceed with image upload and validation
                            processImageUpload();
                        }
                    } else {
                        // No existing image, proceed directly
                        processImageUpload();
                    }
                } else {
                    $error = "❌ لم يتم العثور على دفعة لهذا الرقم. الرجاء التحقق من رقم الطالب أو دفع الرسوم أولاً.";
                    $paymentVerified = false;
                    $step = 1;
                }
                $stmt->close();
            }
        }
    }
} elseif (isset($_GET['step']) && $_GET['step'] == '2' && isset($_GET['student_number'])) {
    // Allow direct access to step 2 with student number in URL
    $step = 2;
    $studentNumber = trim($_GET['student_number']);
    
    // Check payment
    $stmt = $mysqli1->prepare("SELECT student_name FROM payments WHERE student_id = ? LIMIT 1");
    if ($stmt) {
        $stmt->bind_param("s", $studentNumber);
        $stmt->execute();
        $stmt->store_result();
        
        if ($stmt->num_rows > 0) {
            $stmt->bind_result($studentName);
            $stmt->fetch();
            $paymentVerified = true;
            
            // Check if image already exists
            $imageDir = 'uploads/';
            $imagePattern = $imageDir . $studentNumber . '.*';
            $existingFiles = glob($imagePattern);
            
            if (!empty($existingFiles)) {
                $existingImage = $existingFiles[0];
            }
        } else {
            $error = "❌ لم يتم العثور على دفعة لهذا الرقم.";
            $paymentVerified = false;
            $step = 1;
        }
        $stmt->close();
    }
}

// Handle CAPTCHA refresh
if (isset($_GET['refresh_captcha'])) {
    $_SESSION['captcha'] = generateCaptcha();
    exit(json_encode(['captcha' => $_SESSION['captcha']]));
}

function processImageUpload() {
    global $apiUrl, $apiKey, $studentNumber, 
           $studentName, $existingImage, $overwriteImage, $imagePreview, $apiResponse,
           $error, $success, $validationResults, $allValid, $showValidationResults,
           $captchaError;
    
    // Validate manual CAPTCHA
    $userCaptcha = trim($_POST['captcha_input'] ?? '');
    $sessionCaptcha = $_SESSION['captcha'] ?? '';
    
    if (empty($userCaptcha)) {
        $error = "الرجاء إدخال الأرقام التي تراها.";
        $captchaError = true;
        return;
    }
    
    if (strtoupper($userCaptcha) !== strtoupper($sessionCaptcha)) {
        $error = "❌ الأرقام التي أدخلتها غير صحيحة. الرجاء المحاولة مرة أخرى.";
        $captchaError = true;
        
        // Generate new CAPTCHA
        $_SESSION['captcha'] = generateCaptcha();
        return;
    }
    
    // Clear the CAPTCHA after successful validation
    $_SESSION['captcha'] = generateCaptcha();
    
    // Check if file was uploaded without errors
    if (!isset($_FILES['photo']) || $_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
        handleUploadError();
        return;
    }
    
    // Validate file type
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $fileType = mime_content_type($_FILES['photo']['tmp_name']);
    
    if (!in_array($fileType, $allowedTypes)) {
        $error = "نوع الملف غير صالح. يُسمح فقط بصور JPEG و PNG و GIF و WebP.";
        return;
    }
    
    // Validate file size (max 5MB)
    $maxFileSize = 5 * 1024 * 1024; // 5MB
    if ($_FILES['photo']['size'] > $maxFileSize) {
        $error = "الملف كبير جدًا. الحد الأقصى للحجم هو 5 ميجابايت.";
        return;
    }
    
    // Read and convert file to base64
    $imageData = file_get_contents($_FILES['photo']['tmp_name']);
    $base64Image = base64_encode($imageData);
    
    // Prepare for image preview
    $imagePreview = 'data:' . $fileType . ';base64,' . base64_encode($imageData);
    
    // Call API for validation
    if (!callValidationAPI($base64Image)) {
        return; // Error already set in function
    }
    
    // Process validation results
    if (!empty($validationResults)) {
        $allValid = true;
        foreach ($validationResults as $value) {
            if ($value !== true) {
                $allValid = false;
                break;
            }
        }
        
        $showValidationResults = true;
        
        // Only save the image if ALL validations pass
        if ($allValid) {
            saveImageToDisk($_FILES['photo']['tmp_name'], $fileType, $overwriteImage);
        }
    } else {
        $error = "❌ لم يتم استلام نتائج التحقق من الخادم.";
    }
}

function callValidationAPI($base64Image) {
    global $apiUrl, $apiKey, $apiResponse, $error, $validationResults;
    
    // Prepare the JSON data for API
    $jsonData = json_encode([
        'image_base64' => $base64Image
    ]);
    
    // Initialize cURL session
    $ch = curl_init($apiUrl);
    
    // Set cURL options with X-API-Key header
    $headers = [
        'Content-Type: application/json',
        'Content-Length: ' . strlen($jsonData),
        'X-API-Key: ' . $apiKey
    ];
    
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Execute the request
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    // Check for cURL errors
    if (curl_errno($ch)) {
        $error = 'خطأ في الاتصال بالخادم: ' . curl_error($ch);
        curl_close($ch);
        return false;
    }
    
    // Check HTTP status code
    if ($httpCode === 401) {
        $error = 'فشل المصادقة: مفتاح API غير صالح';
        curl_close($ch);
        return false;
    } elseif ($httpCode === 403) {
        $error = 'محظور الوصول: مفتاح API لا يملك الصلاحية';
        curl_close($ch);
        return false;
    } elseif ($httpCode >= 200 && $httpCode < 300) {
        // Parse JSON response
        $apiResponse = json_decode($response, true);
        
        // Check if JSON parsing was successful
        if (json_last_error() !== JSON_ERROR_NONE) {
            $error = 'رد غير صالح من الخادم: ' . json_last_error_msg();
            $apiResponse = null;
            curl_close($ch);
            return false;
        }
        
        // Extract validation results from response
        if (isset($apiResponse['validation'])) {
            $validationResults = $apiResponse['validation'];
        } elseif (isset($apiResponse['faceDetected']) || isset($apiResponse['face_detected'])) {
            // Extract individual validation fields
            $validationFields = [
                'faceDetected', 'face_detected',
                'faceCentered', 'face_centered',
                'appropriateHeadSize', 'appropriate_head_size',
                'whiteBackground', 'white_background'
            ];
            
            foreach ($validationFields as $field) {
                if (isset($apiResponse[$field])) {
                    $normalizedField = str_replace('_', '', lcfirst(ucwords($field, '_')));
                    $validationResults[$normalizedField] = $apiResponse[$field];
                }
            }
        }
    } else {
        $error = "❌ فشل طلب API برمز HTTP: $httpCode";
        curl_close($ch);
        return false;
    }
    
    curl_close($ch);
    return true;
}

function saveImageToDisk($tmpFilePath, $fileType, $overwrite) {
    global $studentNumber, $existingImage, $success, $error;
    
    // Create uploads directory if it doesn't exist
    $uploadDir = 'uploads/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Determine file extension based on MIME type
    $extensions = [
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp'
    ];
    
    $extension = $extensions[$fileType] ?? 'jpg';
    $filename = $studentNumber . '.' . $extension;
    $filepath = $uploadDir . $filename;
    
    // If file exists and we're not overwriting, create new name
    if (!$overwrite && file_exists($filepath)) {
        $counter = 1;
        while (file_exists($filepath)) {
            $filename = $studentNumber . '_' . $counter . '.' . $extension;
            $filepath = $uploadDir . $filename;
            $counter++;
        }
    }
    
    // Delete old file if overwriting
    if ($overwrite && $existingImage && file_exists($existingImage)) {
        unlink($existingImage);
    }
    
    // Save the uploaded file
    if (move_uploaded_file($tmpFilePath, $filepath)) {
        $success = "✅ تم حفظ الصورة بنجاح باسم: " . htmlspecialchars($filename);
        
        // If overwritten, mention it
        if ($overwrite && $existingImage) {
            $success .= " (تم استبدال الصورة السابقة)";
        }
    } else {
        $error = "❌ فشل حفظ الصورة على القرص.";
    }
}

function handleUploadError() {
    global $error;
    $uploadError = $_FILES['photo']['error'] ?? null;
    
    switch ($uploadError) {
        case UPLOAD_ERR_INI_SIZE:
        case UPLOAD_ERR_FORM_SIZE:
            $error = "الملف كبير جدًا.";
            break;
        case UPLOAD_ERR_PARTIAL:
            $error = "تم رفع الملف جزئيًا فقط.";
            break;
        case UPLOAD_ERR_NO_FILE:
            $error = "لم يتم رفع أي ملف.";
            break;
        case UPLOAD_ERR_NO_TMP_DIR:
            $error = "مجلد مؤقت مفقود.";
            break;
        case UPLOAD_ERR_CANT_WRITE:
            $error = "فشل كتابة الملف على القرص.";
            break;
        case UPLOAD_ERR_EXTENSION:
            $error = "توقف رفع الملف بواسطة الامتداد.";
            break;
        default:
            $error = "خطأ غير معروف في الرفع.";
            break;
    }
}

// Function to generate CAPTCHA image - UPDATED VERSION
function generateCaptchaImage($text) {
    // Create image
    $width = 200;
    $height = 60;
    $image = imagecreatetruecolor($width, $height);
    
    // Colors
    $backgroundColor = imagecolorallocate($image, 255, 255, 255);
    $textColor = imagecolorallocate($image, 0, 0, 0);
    $lineColor = imagecolorallocate($image, 200, 200, 200);
    $pixelColor = imagecolorallocate($image, 150, 150, 150);
    
    // Fill background
    imagefilledrectangle($image, 0, 0, $width, $height, $backgroundColor);
    
    // Add lines for noise
    for ($i = 0; $i < 6; $i++) {
        imageline($image, 0, rand() % $height, $width, rand() % $height, $lineColor);
    }
    
    // Add pixels for noise
    for ($i = 0; $i < 100; $i++) {
        imagesetpixel($image, rand() % $width, rand() % $height, $pixelColor);
    }
    
    // Font settings
    $font = dirname(__FILE__) . '/arial.ttf'; // You need to have arial.ttf in your directory
    if (!file_exists($font)) {
        // Use built-in font if arial.ttf doesn't exist
        $fontSize = 5;
        $textLength = strlen($text);
        $charWidth = floor($width / $textLength);
        
        for ($i = 0; $i < $textLength; $i++) {
            $char = $text[$i];
            $x = ($i * $charWidth) + floor($charWidth / 3);
            $y = rand(floor($height / 2), $height - 10);
            
            imagestring($image, $fontSize, $x, $y - 20, $char, $textColor);
        }
    } else {
        // Use TrueType font
        $textLength = strlen($text);
        $charWidth = floor($width / $textLength);
        
        for ($i = 0; $i < $textLength; $i++) {
            $char = $text[$i];
            $x = ($i * $charWidth) + floor($charWidth / 3);
            $y = rand(floor($height / 2), $height - 10);
            $angle = rand(-15, 15);
            $fontSize = rand(24, 28);
            
            imagettftext($image, $fontSize, $angle, $x, $y, $textColor, $font, $char);
        }
    }
    
    // Output image
    ob_start();
    imagepng($image);
    $imageData = ob_get_clean();
    imagedestroy($image);
    
    return 'data:image/png;base64,' . base64_encode($imageData);
}

// Simple CAPTCHA text display (without image generation issues)
$captchaDisplay = $_SESSION['captcha'];

// Translation arrays
$validationLabels = [
    'faceDetected' => 'اكتشاف الوجه',
    'faceCentered' => 'الوجه في المنتصف',
    'appropriateHeadSize' => 'حجم الرأس مناسب',
    'whiteBackground' => 'خلفية بيضاء',
    'imageQuality' => 'جودة الصورة',
    'noGlasses' => 'بدون نظارات',
    'neutralExpression' => 'تعبير محايد',
    'eyesOpen' => 'العينان مفتوحتان',
    'goodLighting' => 'إضاءة جيدة',
    'noHeadwear' => 'بدون غطاء رأس'
];

$validationDescriptions = [
    'faceDetected' => 'تم اكتشاف وجه واضح في الصورة',
    'faceCentered' => 'الوجه موجود في منتصف الصورة بشكل مناسب',
    'appropriateHeadSize' => 'حجم الرأس يشغل النسبة الصحيحة من الصورة',
    'whiteBackground' => 'الخلفية بيضاء نقية ومناسبة',
    'imageQuality' => 'الصورة واضحة وحادة بدون تشويش',
    'noGlasses' => 'الوجه يظهر بدون نظارات',
    'neutralExpression' => 'التعبير الوجهي محايد وطبيعي',
    'eyesOpen' => 'العينان مفتوحتان وواضحتان',
    'goodLighting' => 'الإضاءة موزعة بشكل متساوي على الوجه',
    'noHeadwear' => 'الوجه يظهر بدون أي غطاء للرأس'
];
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نظام رفع صور الطلاب</title>
    <!-- Arabic font -->
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Cairo', sans-serif;
            line-height: 1.8;
            color: #333;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            padding: 20px;
            text-align: right;
            background-attachment: fixed;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #1a642b 0%, #2c3e50 100%);
            color: white;
            padding: 40px 20px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 15px;
        }
        
        .header h1 i {
            color: #f9dd10;
            margin-left: 15px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
        }
        
        /* Progress Steps */
        .progress-steps {
            display: flex;
            justify-content: center;
            padding: 20px;
            background: #f8f9fa;
            border-bottom: 1px solid #dee2e6;
        }
        
        .step {
            display: flex;
            align-items: center;
            margin: 0 10px;
        }
        
        .step-number {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #dee2e6;
            color: #6c757d;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            margin-left: 10px;
        }
        
        .step.active .step-number {
            background: #1a642b;
            color: white;
        }
        
        .step.completed .step-number {
            background: #4caf50;
            color: white;
        }
        
        .step-text {
            font-weight: 600;
            color: #6c757d;
        }
        
        .step.active .step-text {
            color: #1a642b;
        }
        
        .step.completed .step-text {
            color: #4caf50;
        }
        
        .step-line {
            width: 100px;
            height: 3px;
            background: #dee2e6;
            margin: 0 10px;
        }
        
        .step.active ~ .step-line {
            background: #dee2e6;
        }
        
        .step.active .step-line,
        .step.completed .step-line {
            background: #1a642b;
        }
        
        .content {
            padding: 30px;
        }
        
        /* Form Container */
        .form-container {
            max-width: 800px;
            margin: 0 auto;
        }
        
        /* Student Info Box */
        .student-info-box {
            background: #e8f5e9;
            border: 2px solid #4caf50;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
        }
        
        .student-info-box.invalid {
            background: #ffebee;
            border-color: #f44336;
        }
        
        .student-info-header {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .student-info-icon {
            font-size: 2.5rem;
        }
        
        .student-info-title {
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .student-info-details {
            display: flex;
            justify-content: center;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .info-item {
            background: white;
            padding: 15px 25px;
            border-radius: 8px;
            border: 1px solid #ddd;
            min-width: 200px;
            text-align: center;
        }
        
        .info-label {
            font-size: 0.9rem;
            color: #666;
            margin-bottom: 5px;
        }
        
        .info-value {
            font-size: 1.3rem;
            font-weight: 700;
            color: #1a642b;
        }
        
        /* Existing Image Warning */
        .existing-image-warning {
            background: #fff3cd;
            border: 2px solid #ffc107;
            border-radius: 10px;
            padding: 25px;
            margin: 25px 0;
            text-align: center;
        }
        
        .existing-image-warning h3 {
            color: #856404;
            font-size: 1.5rem;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .existing-image-preview {
            max-width: 200px;
            margin: 20px auto;
            border: 3px solid #ffc107;
            border-radius: 8px;
            overflow: hidden;
        }
        
        .existing-image-preview img {
            width: 100%;
            height: auto;
        }
        
        .overwrite-options {
            display: flex;
            justify-content: center;
            gap: 20px;
            margin-top: 25px;
            flex-wrap: wrap;
        }
        
        .overwrite-option {
            padding: 15px 30px;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            border: 2px solid;
            background: white;
            transition: all 0.3s ease;
        }
        
        .overwrite-yes {
            color: #856404;
            border-color: #ffc107;
        }
        
        .overwrite-yes:hover {
            background: #ffc107;
            color: white;
        }
        
        .overwrite-no {
            color: #721c24;
            border-color: #f5c6cb;
        }
        
        .overwrite-no:hover {
            background: #f5c6cb;
            color: #721c24;
        }
        
        /* Upload Form */
        .upload-form {
            margin-top: 30px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #1a642b;
            font-size: 1.1rem;
        }
        
        .form-group input[type="text"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 1rem;
            font-family: 'Cairo', sans-serif;
            transition: all 0.3s ease;
        }
        
        .form-group input[type="text"]:focus {
            outline: none;
            border-color: #1a642b;
            box-shadow: 0 0 0 3px rgba(26, 100, 43, 0.1);
        }
        
        /* File Input */
        .file-input-wrapper {
            position: relative;
            margin: 20px 0;
        }
        
        .file-input {
            width: 100%;
            padding: 20px;
            border: 2px dashed #1a642b;
            border-radius: 10px;
            background: rgba(26, 100, 43, 0.05);
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            font-family: 'Cairo', sans-serif;
            font-size: 1rem;
        }
        
        .file-input:hover {
            border-color: #f9dd10;
            background: rgba(249, 221, 16, 0.1);
        }
        
        /* Manual CAPTCHA */
        .captcha-container {
            background: rgba(249, 221, 16, 0.1);
            padding: 20px;
            border-radius: 10px;
            border: 2px solid #f9dd10;
            margin: 20px 0;
            text-align: center;
        }
        
        .captcha-instruction {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            margin-bottom: 15px;
            color: #333;
            font-size: 1.1rem;
        }
        
        .captcha-display {
            background: white;
            padding: 15px;
            border-radius: 8px;
            border: 2px solid #ddd;
            margin: 15px auto;
            display: inline-block;
        }
        
        .captcha-image {
            font-family: 'Courier New', monospace;
            font-size: 28px;
            font-weight: bold;
            letter-spacing: 10px;
            color: #1a642b;
            background: linear-gradient(45deg, #f8f9fa, #e9ecef);
            padding: 15px 25px;
            border-radius: 5px;
            border: 2px solid #dee2e6;
            text-align: center;
            direction: ltr;
            user-select: none;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
        }
        
        .captcha-input-group {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 15px;
            margin-top: 20px;
        }
        
        .captcha-input {
            width: 220px;
            padding: 12px;
            border: 2px solid #1a642b;
            border-radius: 8px;
            font-size: 1.2rem;
            font-family: 'Courier New', monospace;
            text-align: center;
            letter-spacing: 5px;
            direction: ltr;
        }
        
        .captcha-input:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(26, 100, 43, 0.2);
        }
        
        .refresh-captcha {
            background: #1a642b;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: background 0.3s ease;
        }
        
        .refresh-captcha:hover {
            background: #2c3e50;
        }
        
        .captcha-info {
            color: #666;
            font-size: 0.9rem;
            margin-top: 15px;
        }
        
        /* Buttons */
        .form-buttons {
            display: flex;
            justify-content: space-between;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #dee2e6;
        }
        
        .btn {
            padding: 12px 30px;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            border: 2px solid;
            transition: all 0.3s ease;
            font-family: 'Cairo', sans-serif;
        }
        
        .btn-prev {
            background: white;
            color: #6c757d;
            border-color: #6c757d;
        }
        
        .btn-prev:hover {
            background: #6c757d;
            color: white;
        }
        
        .btn-next, .btn-submit {
            background: #1a642b;
            color: white;
            border-color: #1a642b;
        }
        
        .btn-next:hover:not(:disabled),
        .btn-submit:hover:not(:disabled) {
            background: #2c3e50;
            border-color: #2c3e50;
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Error/Success Messages */
        .error {
            background: #ffebee;
            border-right: 4px solid #f44336;
            padding: 20px;
            margin: 20px 0;
            border-radius: 10px;
            text-align: right;
            font-size: 1rem;
        }
        
        .success {
            background: #e8f5e9;
            border-right: 4px solid #4caf50;
            padding: 20px;
            margin: 20px 0;
            border-radius: 10px;
            text-align: right;
            font-size: 1rem;
        }
        
        /* Validation Results */
        .validation-results {
            margin-top: 30px;
            padding: 25px;
            background: #f8f9fa;
            border-radius: 15px;
            border: 2px solid #dee2e6;
        }
        
        .validation-header {
            text-align: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #6c757d;
        }
        
        .validation-header h3 {
            font-size: 1.8rem;
            color: #1a642b;
            margin-bottom: 10px;
        }
        
        .validation-status {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            margin-bottom: 20px;
            padding: 15px;
            border-radius: 10px;
            font-size: 1.2rem;
            font-weight: 600;
        }
        
        .validation-status.success {
            background: #e8f5e9;
            color: #1a642b;
            border: 2px solid #4caf50;
        }
        
        .validation-status.error {
            background: #ffebee;
            color: #f44336;
            border: 2px solid #f44336;
        }
        
        .validation-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 25px 0;
        }
        
        .validation-item {
            background: white;
            padding: 20px;
            border-radius: 10px;
            border: 2px solid #ddd;
            display: flex;
            align-items: center;
            gap: 15px;
            transition: all 0.3s ease;
        }
        
        .validation-item.valid {
            border-color: #4caf50;
            background: rgba(76, 175, 80, 0.05);
        }
        
        .validation-item.invalid {
            border-color: #f44336;
            background: rgba(244, 67, 54, 0.05);
        }
        
        .validation-icon {
            font-size: 1.8rem;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            flex-shrink: 0;
        }
        
        .validation-item.valid .validation-icon {
            background: #4caf50;
            color: white;
        }
        
        .validation-item.invalid .validation-icon {
            background: #f44336;
            color: white;
        }
        
        .validation-details {
            flex: 1;
        }
        
        .validation-title {
            font-weight: 700;
            font-size: 1.1rem;
            margin-bottom: 5px;
            color: #333;
        }
        
        .validation-description {
            color: #666;
            font-size: 0.9rem;
        }
        
        /* Requirements Section */
        .requirements-box {
            background: #fff8e1;
            border: 2px solid #ffc107;
            border-radius: 10px;
            padding: 25px;
            margin: 25px 0;
        }
        
        .requirements-box h4 {
            color: #ff9800;
            font-size: 1.4rem;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .requirements-list {
            margin-right: 20px;
        }
        
        .requirements-list li {
            margin-bottom: 12px;
            padding-right: 10px;
            color: #5a6268;
            line-height: 1.6;
        }
        
        /* Action Buttons */
        .action-buttons {
            display: flex;
            justify-content: center;
            gap: 20px;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 2px solid #dee2e6;
        }
        
        .action-btn {
            padding: 15px 30px;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            transition: all 0.3s ease;
            cursor: pointer;
            border: 2px solid;
        }
        
        .reset-btn {
            background: #ffc107;
            color: #333;
            border-color: #ff9800;
        }
        
        .reset-btn:hover {
            background: #ff9800;
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(255, 152, 0, 0.2);
        }
        
        .upload-new-btn {
            background: #1a642b;
            color: white;
            border-color: #1a642b;
        }
        
        .upload-new-btn:hover {
            background: #2c3e50;
            border-color: #2c3e50;
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(26, 100, 43, 0.2);
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .container {
                margin: 10px;
                border-radius: 10px;
            }
            
            .content {
                padding: 20px;
            }
            
            .header h1 {
                font-size: 2rem;
            }
            
            .progress-steps {
                flex-direction: column;
                align-items: center;
                gap: 15px;
            }
            
            .step {
                width: 100%;
                justify-content: center;
            }
            
            .step-line {
                display: none;
            }
            
            .student-info-details {
                flex-direction: column;
            }
            
            .info-item {
                width: 100%;
            }
            
            .validation-grid {
                grid-template-columns: 1fr;
            }
            
            .form-buttons, .action-buttons, .overwrite-options {
                flex-direction: column;
            }
            
            .btn, .action-btn, .overwrite-option {
                width: 100%;
                text-align: center;
            }
            
            .captcha-image {
                font-size: 20px;
                letter-spacing: 6px;
                padding: 10px 15px;
            }
            
            .captcha-input {
                width: 180px;
                font-size: 1rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-graduation-cap"></i> نظام رفع صور الطلاب</h1>
            <p>رفع صور الطلاب مع التحقق الآلي والتسمية التلقائية</p>
        </div>
        
        <!-- Progress Steps -->
        <div class="progress-steps">
            <div class="step <?php echo $step >= 1 ? 'completed' : ($step == 1 ? 'active' : ''); ?>">
                <div class="step-number">1</div>
                <div class="step-text">التحقق من الطالب</div>
            </div>
            <div class="step-line"></div>
            <div class="step <?php echo $step >= 2 ? 'completed' : ($step == 2 ? 'active' : ''); ?>">
                <div class="step-number">2</div>
                <div class="step-text">رفع الصورة</div>
            </div>
        </div>
        
        <div class="content">
            <!-- System Status -->
            <?php if (empty($apiKey)): ?>
                <div class="error">
                    <strong>⚠️ خطأ في إعداد النظام</strong>
                    <p>مفتاح API غير مضبوط. يرجى الاتصال بالمسؤول.</p>
                </div>
            <?php endif; ?>
            
            <!-- Error Display -->
            <?php if ($error): ?>
                <div class="error">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span style="font-size: 2rem; color: #f44336;">❌</span>
                        <div>
                            <h4 style="margin: 0 0 10px 0; color: #f44336;">خطأ</h4>
                            <p style="margin: 0;"><?php echo htmlspecialchars($error); ?></p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Success Message -->
            <?php if ($success): ?>
                <div class="success">
                    <div style="display: flex; align-items: center; gap: 15px;">
                        <span style="font-size: 2rem; color: #4caf50;">✅</span>
                        <div>
                            <h4 style="margin: 0 0 10px 0; color: #4caf50;">تم بنجاح!</h4>
                            <p style="margin: 0;"><?php echo htmlspecialchars($success); ?></p>
                        </div>
                    </div>
                    <div class="action-buttons">
                        <button onclick="window.location.reload()" class="action-btn upload-new-btn">
                            <i class="fas fa-plus"></i> رفع صورة جديدة
                        </button>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Student Info Box -->
            <?php if ($step == 2 && $paymentVerified): ?>
                <div class="student-info-box">
                    <div class="student-info-header">
                        <div class="student-info-icon">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <div class="student-info-title">
                            ✅ تم التحقق من الطالب
                        </div>
                    </div>
                    
                    <div class="student-info-details">
                        <div class="info-item">
                            <div class="info-label">رقم الطالب</div>
                            <div class="info-value"><?php echo htmlspecialchars($studentNumber); ?></div>
                        </div>
                        
                        <?php if (!empty($studentName)): ?>
                        <div class="info-item">
                            <div class="info-label">اسم الطالب</div>
                            <div class="info-value"><?php echo htmlspecialchars($studentName); ?></div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($existingImage): ?>
                        <div class="info-item">
                            <div class="info-label">حالة الصورة</div>
                            <div class="info-value" style="color: #ff9800;">موجودة سابقاً</div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Step 1: Student Check Form -->
            <?php if ($step == 1): ?>
                <div class="form-container">
                    <form method="POST" class="upload-form" id="studentForm">
                        <input type="hidden" name="step" value="1">
                        
                        <div class="form-group">
                            <label for="student_number">
                                <i class="fas fa-id-card"></i> أدخل رقم الطالب للتحقق *
                            </label>
                            <input type="text" 
                                   id="student_number" 
                                   name="student_number" 
                                   value="<?php echo htmlspecialchars($studentNumber); ?>"
                                   placeholder="أدخل رقم الطالب"
                                   required
                                   <?php echo empty($apiKey) ? 'disabled' : ''; ?>>
                            <small style="color: #666; display: block; margin-top: 5px;">
                                سيتم التحقق من وجود دفعة لهذا الرقم قبل المتابعة
                            </small>
                        </div>
                        
                        <div class="form-buttons">
                            <button type="submit" class="btn btn-next" id="submitBtn">
                                <i class="fas fa-arrow-left"></i> التالي: التحقق من الطالب
                            </button>
                        </div>
                    </form>
                </div>
            <?php endif; ?>
            
            <!-- Step 2: Image Upload Form -->
            <?php if ($step == 2 && $paymentVerified && !$showValidationResults): ?>
                <div class="form-container">
                    <?php if ($existingImage && !$overwriteImage): ?>
                        <!-- Existing Image Warning -->
                        <div class="existing-image-warning">
                            <h3>
                                <i class="fas fa-exclamation-triangle"></i>
                                تنبيه: هناك صورة موجودة مسبقاً
                            </h3>
                            <p>
                                تم العثور على صورة مرفوعة سابقاً للطالب 
                                <strong><?php echo htmlspecialchars($studentNumber); ?></strong>
                                <br>
                                هل تريد استبدال الصورة الحالية؟
                            </p>
                            
                            <?php if ($existingImage): ?>
                            <div class="existing-image-preview">
                                <img src="<?php echo htmlspecialchars($existingImage); ?>" 
                                     alt="الصورة الحالية"
                                     onerror="this.style.display='none'">
                            </div>
                            <?php endif; ?>
                            
                            <div class="overwrite-options">
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="step" value="2">
                                    <input type="hidden" name="student_number" value="<?php echo htmlspecialchars($studentNumber); ?>">
                                    <input type="hidden" name="overwrite_image" value="1">
                                    <button type="submit" class="overwrite-option overwrite-yes">
                                        <i class="fas fa-check"></i> نعم، استبدال الصورة
                                    </button>
                                </form>
                                
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="step" value="2">
                                    <input type="hidden" name="student_number" value="<?php echo htmlspecialchars($studentNumber); ?>">
                                    <input type="hidden" name="overwrite_image" value="0">
                                    <button type="submit" class="overwrite-option overwrite-no">
                                        <i class="fas fa-times"></i> لا، إنشاء صورة جديدة
                                    </button>
                                </form>
                            </div>
                            
                            <div style="margin-top: 20px; color: #666; font-size: 0.9rem;">
                                <i class="fas fa-info-circle"></i>
                                إذا اخترت "إنشاء صورة جديدة"، سيتم حفظ الصورة باسم جديد مع إضافة رقم تسلسلي
                            </div>
                        </div>
                    <?php else: ?>
                        <!-- Image Upload Form -->
                        <form method="POST" enctype="multipart/form-data" class="upload-form" id="uploadForm">
                            <input type="hidden" name="step" value="2">
                            <input type="hidden" name="student_number" value="<?php echo htmlspecialchars($studentNumber); ?>">
                            <input type="hidden" name="overwrite_image" value="<?php echo $overwriteImage ? '1' : '0'; ?>">
                            
                            <div class="form-group">
                                <label for="photo">
                                    <i class="fas fa-camera"></i> اختر صورة الطالب *
                                </label>
                                <div class="file-input-wrapper">
                                    <input type="file" 
                                           name="photo" 
                                           id="photo" 
                                           accept="image/*" 
                                           required 
                                           class="file-input"
                                           onchange="previewImage(this)">
                                </div>
                                <small style="color: #666; display: block; margin-top: 5px;">
                                    يُسمح بصور JPEG، PNG، GIF، WebP بحد أقصى 5 ميجابايت
                                </small>
                            </div>
                            
                            <!-- Image Preview -->
                            <div id="imagePreviewContainer" style="display: none; margin: 20px 0;">
                                <div style="text-align: center; padding: 20px; border: 2px dashed #1a642b; border-radius: 10px;">
                                    <img id="previewImage" src="" alt="معاينة الصورة" style="max-width: 300px; max-height: 300px; border-radius: 8px;">
                                    <div id="imageInfo" style="margin-top: 10px; color: #666;"></div>
                                </div>
                            </div>
                            
                            <!-- Manual CAPTCHA -->
                            <div class="captcha-container">
                                <div class="captcha-instruction">
                                    <i class="fas fa-shield-alt"></i>
                                    <span>الرجاء التحقق من أنك لست روبوتًا</span>
                                </div>
                                
                                <div class="captcha-display">
                                    <div class="captcha-image" id="captchaText">
                                        <?php echo htmlspecialchars($captchaDisplay); ?>
                                    </div>
                                </div>
                                
                                <div class="captcha-input-group">
                                    <input type="text" 
                                           name="captcha_input" 
                                           id="captcha_input" 
                                           class="captcha-input"
                                           placeholder="اكتب الأرقام كما تراها"
                                           maxlength="6"
                                           autocomplete="off"
                                           dir="ltr"
                                           <?php echo $captchaError ? 'style="border-color: #f44336;"' : ''; ?>>
                                    <button type="button" class="refresh-captcha" onclick="refreshCaptcha()">
                                        <i class="fas fa-sync-alt"></i> تحديث الأرقام
                                    </button>
                                </div>
                                
                                <div class="captcha-info">
                                    <i class="fas fa-info-circle"></i>
                                    هذا التحقق يضمن أمان النظام ويمنع الاستخدام الآلي
                                </div>
                            </div>
                            
                            <div class="form-buttons">
                                <button type="button" class="btn btn-prev" onclick="goBackToStep1()">
                                    <i class="fas fa-arrow-right"></i> رجوع
                                </button>
                                <button type="submit" class="btn btn-submit" id="submitBtn">
                                    <i class="fas fa-upload"></i> رفع وتحقق من الصورة
                                </button>
                            </div>
                        </form>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <!-- Validation Results -->
            <?php if ($showValidationResults && !empty($validationResults)): 
                $validCount = 0;
                $totalCount = count($validationResults);
                
                foreach ($validationResults as $value) {
                    if ($value === true) {
                        $validCount++;
                    }
                }
                
                $successPercentage = $totalCount > 0 ? round(($validCount / $totalCount) * 100) : 0;
            ?>
                <div class="validation-results" id="validationResults">
                    <div class="validation-header">
                        <h3><i class="fas fa-clipboard-check"></i> نتائج التحقق</h3>
                        <div class="validation-status <?php echo $allValid ? 'success' : 'error'; ?>">
                            <?php if ($allValid): ?>
                                <i class="fas fa-check-circle"></i>
                                <span>✅ تم التحقق بنجاح - جميع المعايير مطابقة</span>
                            <?php else: ?>
                                <i class="fas fa-exclamation-circle"></i>
                                <span>⚠️ الصورة غير مطابقة للمعايير - النسبة: <?php echo $successPercentage; ?>%</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Validation Details -->
                    <div class="validation-grid">
                        <?php foreach ($validationResults as $key => $value): 
                            $isValid = $value === true;
                            $arabicLabel = $validationLabels[$key] ?? $key;
                            $arabicDescription = $validationDescriptions[$key] ?? 'معيار التحقق';
                        ?>
                            <div class="validation-item <?php echo $isValid ? 'valid' : 'invalid'; ?>">
                                <div class="validation-icon">
                                    <?php if ($isValid): ?>
                                        <i class="fas fa-check"></i>
                                    <?php else: ?>
                                        <i class="fas fa-times"></i>
                                    <?php endif; ?>
                                </div>
                                <div class="validation-details">
                                    <div class="validation-title"><?php echo htmlspecialchars($arabicLabel); ?></div>
                                    <div class="validation-description"><?php echo htmlspecialchars($arabicDescription); ?></div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <?php if (!$allValid): ?>
                        <!-- Requirements Section -->
                        <div class="requirements-box">
                            <h4><i class="fas fa-exclamation-triangle"></i> متطلبات الصورة الشخصية</h4>
                            <ul class="requirements-list">
                                <li><strong>وجه واضح:</strong> يجب أن يظهر الوجه بوضوح في الصورة</li>
                                <li><strong>مركزية:</strong> الوجه يجب أن يكون في منتصف الصورة</li>
                                <li><strong>حجم مناسب:</strong> حجم الرأس يجب أن يشغل حوالي 60-70% من الصورة</li>
                                <li><strong>خلفية بيضاء:</strong> يجب أن تكون الخلفية بيضاء نقية بدون أنماط</li>
                                <li><strong>جودة عالية:</strong> الصورة يجب أن تكون واضحة وحادة</li>
                                <li><strong>بدون نظارات:</strong> الوجه يجب أن يظهر بدون نظارات</li>
                                <li><strong>تعبير محايد:</strong> التعبير الوجهي يجب أن يكون طبيعياً ومحايداً</li>
                                <li><strong>عيون مفتوحة:</strong> العينان يجب أن تكونا مفتوحتين وواضحتين</li>
                                <li><strong>إضاءة جيدة:</strong> إضاءة موزعة بشكل متساوي على الوجه</li>
                                <li><strong>بدون غطاء رأس:</strong> الوجه يجب أن يظهر بدون أي غطاء للرأس</li>
                            </ul>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <?php if (!$allValid): ?>
                            <button onclick="goBackToStep2()" class="action-btn reset-btn">
                                <i class="fas fa-sync-alt"></i> رفع صورة جديدة
                            </button>
                            <button onclick="window.location.reload()" class="action-btn upload-new-btn">
                                <i class="fas fa-redo"></i> إعادة تحميل الصفحة
                            </button>
                        <?php else: ?>
                            <button onclick="window.location.reload()" class="action-btn upload-new-btn">
                                <i class="fas fa-plus"></i> رفع صورة جديدة
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Image preview functionality
        function previewImage(input) {
            const previewContainer = document.getElementById('imagePreviewContainer');
            const preview = document.getElementById('previewImage');
            const imageInfo = document.getElementById('imageInfo');
            
            if (input.files && input.files[0]) {
                const file = input.files[0];
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    // Show preview
                    preview.src = e.target.result;
                    previewContainer.style.display = 'block';
                    
                    // Clear previous info
                    imageInfo.innerHTML = '';
                    
                    // Create image stats in Arabic
                    const fileSize = (file.size / 1024 / 1024).toFixed(2);
                    const fileType = file.type.split('/')[1]?.toUpperCase() || 'غير معروف';
                    
                    const img = new Image();
                    img.onload = function() {
                        const dimensionsText = `${this.width} × ${this.height} بكسل`;
                        
                        const infoHtml = `
                            <div>الحجم: ${fileSize} ميجابايت</div>
                            <div>النوع: ${fileType}</div>
                            <div>الأبعاد: ${dimensionsText}</div>
                        `;
                        imageInfo.innerHTML = infoHtml;
                    };
                    img.src = e.target.result;
                };
                
                reader.readAsDataURL(file);
            } else {
                // Hide preview
                previewContainer.style.display = 'none';
                imageInfo.innerHTML = '';
            }
        }
        
        // Go back to step 1
        function goBackToStep1() {
            window.location.href = window.location.pathname;
        }
        
        // Go back to step 2
        function goBackToStep2() {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '';
            
            const stepInput = document.createElement('input');
            stepInput.type = 'hidden';
            stepInput.name = 'step';
            stepInput.value = '2';
            form.appendChild(stepInput);
            
            const studentInput = document.createElement('input');
            studentInput.type = 'hidden';
            studentInput.name = 'student_number';
            studentInput.value = '<?php echo htmlspecialchars($studentNumber); ?>';
            form.appendChild(studentInput);
            
            const overwriteInput = document.createElement('input');
            overwriteInput.type = 'hidden';
            overwriteInput.name = 'overwrite_image';
            overwriteInput.value = '0';
            form.appendChild(overwriteInput);
            
            document.body.appendChild(form);
            form.submit();
        }
        
        // Refresh CAPTCHA
        function refreshCaptcha() {
            fetch('?refresh_captcha=1')
                .then(response => response.json())
                .then(data => {
                    if (data.captcha) {
                        document.getElementById('captchaText').textContent = data.captcha;
                        document.getElementById('captcha_input').value = '';
                        document.getElementById('captcha_input').focus();
                    }
                })
                .catch(error => {
                    console.error('Error refreshing CAPTCHA:', error);
                    // Reload page as fallback
                    window.location.reload();
                });
        }
        
        // Focus on CAPTCHA input when page loads
        document.addEventListener('DOMContentLoaded', function() {
            const captchaInput = document.getElementById('captcha_input');
            if (captchaInput) {
                captchaInput.focus();
                
                // Auto-uppercase CAPTCHA input
                captchaInput.addEventListener('input', function() {
                    this.value = this.value.toUpperCase();
                });
            }
            
            // Step 2 form submission
            const uploadForm = document.getElementById('uploadForm');
            if (uploadForm) {
                uploadForm.addEventListener('submit', function(e) {
                    const fileInput = document.getElementById('photo');
                    const captchaInput = document.getElementById('captcha_input');
                    const submitBtn = document.getElementById('submitBtn');
                    
                    if (!fileInput.files || fileInput.files.length === 0) {
                        alert('الرجاء اختيار صورة.');
                        e.preventDefault();
                        return;
                    }
                    
                    if (!captchaInput || !captchaInput.value.trim()) {
                        alert('الرجاء إدخال الأرقام التي تراها.');
                        e.preventDefault();
                        captchaInput.focus();
                        return;
                    }
                    
                    // Show loading state
                    if (submitBtn) {
                        submitBtn.disabled = true;
                        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري المعالجة...';
                    }
                });
            }
        });
    </script>
</body>
</html>