<?php
// /admission33333/admin_login.php — Admin login page for senior management only
// نسخة محسنة بالكامل مع الحماية من SQL Injection والثغرات الأمنية

/* ===== Debug toggle: add ?debug=1 to see errors on page (للتطوير فقط) ===== */
$DEBUG = isset($_GET['debug']) && $_GET['debug'] === '1';
if ($DEBUG) {
  // في بيئة التطوير فقط - إظهار الأخطاء
  ini_set('display_errors', 1);
  ini_set('display_startup_errors', 1);
  error_reporting(E_ALL);
  header("Content-Type: text/html; charset=UTF-8");
} else {
  // في الإنتاج - لا تظهر الأخطاء، فقط سجلها
  ini_set('display_errors', 0);
  ini_set('display_startup_errors', 0);
  error_reporting(E_ALL);
}

/* ===== Set role for admin access ===== */
$role = 'admin'; // Fixed role for admin access

/* ===== Helper to locate required files no matter current dir ===== */
function req_path($fname) {
  $candidates = [
    __DIR__ . "/$fname",
    dirname(__DIR__) . "/$fname",
    __DIR__ . "/../$fname",
  ];
  foreach ($candidates as $p) {
    if (is_file($p)) return $p;
  }
  return null;
}

/* ===== Include db + helpers safely ===== */
$dbphp = req_path('db.php');
$helphp = req_path('helpers.php');

if (!$dbphp || !$helphp) {
  http_response_code(500);
  error_log("CRITICAL: db.php or helpers.php not found in " . __DIR__);
  if ($DEBUG) {
    echo "<h3>خطأ في النظام</h3>";
    echo "<p>الملفات الأساسية غير موجودة: " . htmlspecialchars(__DIR__) . "</p>";
  } else {
    echo "<!DOCTYPE html><html><head><meta charset='UTF-8'><title>خطأ</title></head><body>";
    echo "<h3>خطأ في النظام</h3><p>حدث خطأ داخلي. الرجاء المحاولة لاحقاً.</p>";
    echo "</body></html>";
  }
  exit;
}

require_once $dbphp;
require_once $helphp;

/* ===== Ensure session ===== */
if (function_exists('session_status') && session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

/* ===== دوال الأمان الأساسية (إذا لم تكن موجودة في helpers) ===== */
if (!function_exists('verify_csrf') || !function_exists('csrf_token')) {
  function csrf_token() { 
    return $_SESSION['csrf'] = $_SESSION['csrf'] ?? bin2hex(random_bytes(16)); 
  }
  function verify_csrf($t) { 
    return isset($_SESSION['csrf']) && hash_equals($_SESSION['csrf'], (string)$t); 
  }
}
if (!function_exists('e')) {
  function e($s) { 
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); 
  }
}

/* ===== دوال التحقق من صحة المدخلات ===== */
function validateUsername($value) {
    if ($value === null) return '';
    $value = trim((string)$value);
    // إزالة الأحرف الخطيرة
    $value = preg_replace('/[<>"\']/', '', $value);
    // تحديد طول أقصى 100 حرف
    $value = substr($value, 0, 100);
    return $value;
}

function validatePassword($value) {
    if ($value === null) return '';
    $value = (string)$value;
    // تحديد طول أقصى معقول لكلمة المرور (منع هجمات DoS)
    if (strlen($value) > 255) {
        $value = substr($value, 0, 255);
    }
    return $value;
}

function validateInt($value, $default = null) {
    if ($value === null || $value === '') return $default;
    return filter_var($value, FILTER_VALIDATE_INT) !== false ? (int)$value : $default;
}

function tableExists($table) {
    global $mysqli1;
    try {
        $result = $mysqli1->query("SHOW TABLES LIKE '" . $mysqli1->real_escape_string($table) . "'");
        return $result && $result->num_rows > 0;
    } catch (Exception $e) {
        error_log("Error checking table existence: " . $e->getMessage());
        return false;
    }
}

/* ===== التحقق من اتصال قاعدة البيانات ===== */
if (!isset($mysqli1) || !($mysqli1 instanceof mysqli)) {
  http_response_code(500);
  error_log("CRITICAL: Database connection failed in admin_login.php");
  if ($DEBUG) {
    echo "<h3>خطأ في قاعدة البيانات</h3><p>تعذر الاتصال بقاعدة البيانات.</p>";
  } else {
    echo "<!DOCTYPE html><html><head><meta charset='UTF-8'><title>خطأ</title></head><body>";
    echo "<h3>خطأ في النظام</h3><p>حدث خطأ داخلي. الرجاء المحاولة لاحقاً.</p>";
    echo "</body></html>";
  }
  exit;
}

$mysqli1->set_charset('utf8mb4');
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

/* ===== التحقق من وجود جدول committee_users ===== */
$tableExists = tableExists('committee_users');
if (!$tableExists) {
    error_log("CRITICAL: committee_users table does not exist in database");
}

/* ===== If already logged in as admin, go to admin dashboard ===== */
if (!empty($_SESSION['committee_id'])) {
  $user_role_id = validateInt($_SESSION['committee_role_id'] ?? 0);
  $user_role = (string)($_SESSION['user_role'] ?? '');
  
  // فقط إذا كان المستخدم هو admin ومتصل بجلسة admin
  if ($user_role_id === 1 && $user_role === 'admin') {
    header("Location: admin_dashboard.php");
    exit;
  }
  
  // إذا كان المستخدم لديه committee_id ولكن ليس admin، تدمير الجلسة
  // لأن هذه الصفحة مخصصة للإدارة العليا فقط
  session_destroy();
  $_SESSION = array();
  
  // إذا كان هناك كوكيز للجلسة، احذفها
  if (ini_get("session.use_cookies")) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
      $params["path"], $params["domain"],
      $params["secure"], $params["httponly"]
    );
  }
}

$error = '';
$username = '';

/* ===== Handle POST (login) ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    // التحقق من CSRF token
    if (!verify_csrf($_POST['csrf'] ?? '')) {
      $error = 'رمز الجلسة غير صالح. الرجاء تحديث الصفحة والمحاولة مرة أخرى.';
      error_log("CSRF validation failed in admin_login");
    } else {
      // تنظيف المدخلات
      $username = validateUsername($_POST['username'] ?? '');
      $password = validatePassword($_POST['password'] ?? '');

      // التحقق من الحقول المطلوبة
      if ($username === '') {
        $error = 'اسم المستخدم مطلوب.';
      } elseif ($password === '') {
        $error = 'كلمة المرور مطلوبة.';
      } elseif (!$tableExists) {
        // إذا كان الجدول غير موجود
        error_log("Login attempt but committee_users table missing");
        $error = 'خطأ في النظام. الرجاء الاتصال بالدعم الفني.';
      } else {
        // استخدام prepared statement (آمن تماماً من SQL Injection)
        $stmt = $mysqli1->prepare("SELECT id, username, password_hash, name, can_view_all, college_id, role_id FROM committee_users WHERE username=? AND role_id=1 LIMIT 1");
        if (!$stmt) {
            throw new Exception("Failed to prepare statement: " . $mysqli1->error);
        }
        
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $user = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        // تسجيل محاولة الدخول للأمان
        error_log("Admin login attempt for username: '" . $username . "' - " . ($user ? 'User found' : 'User not found'));

        $ok = false;
        if ($user) {
          // استخدام password_verify فقط (آمن)
          if (password_verify($password, $user['password_hash'])) {
            $ok = true;
            error_log("Password verification SUCCESS for admin: " . $username);
          } else {
            // للأمان، سجل محاولة استخدام كلمة مرور خاطئة
            error_log("Password verification FAILED for admin: " . $username);
          }
        }

        if ($ok) {
          $user_role_id = isset($user['role_id']) ? validateInt($user['role_id']) : 0;
          
          if ($user_role_id !== 1) {
            $error = 'ليس لديك صلاحية للوصول إلى لوحة تحكم الإدارة العليا.';
            error_log("ADMIN ACCESS DENIED: User role_id=$user_role_id attempted to access admin dashboard");
          } else {
            // تجديد معرف الجلسة (آمن)
            session_regenerate_id(true);
            
            // تخزين جميع بيانات المستخدم في الجلسة
            $_SESSION['committee_id']          = validateInt($user['id']);
            $_SESSION['committee_username']    = (string)$user['username'];
            $_SESSION['committee_name']        = (string)$user['name'];
            $_SESSION['committee_can_all']     = validateInt($user['can_view_all']) === 1;
            $_SESSION['committee_college_id']  = isset($user['college_id']) ? validateInt($user['college_id']) : null;
            $_SESSION['committee_role_id']     = $user_role_id;
            
            // Mark session as admin - THIS IS CRITICAL
            $_SESSION['user_role'] = 'admin';
            $_SESSION['is_admin'] = true;

            // تسجيل دخول ناجح
            error_log("Successful admin login for: " . $username . " (ID: " . $user['id'] . ")");

            // Redirect to admin dashboard
            header("Location: admin_dashboard.php");
            exit;
          }
        } else {
          if ($user) {
            $error = 'كلمة المرور غير صحيحة.';
          } else {
            $error = 'اسم المستخدم غير موجود أو ليس لديك صلاحية دخول كإدارة عليا.';
          }
          
          // في الإنتاج، لا تقدم تفاصيل إضافية
          if ($DEBUG) {
            $error .= ' تحقق من سجلات الأخطاء للحصول على معلومات تصحيح مفصلة.';
          }
        }
      }
    }
  } catch (Throwable $e) {
    error_log("[admin_login][EXCEPTION] " . $e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine());
    $error = $DEBUG ? ('خطأ في النظام: ' . $e->getMessage()) : 'خطأ في الخادم. الرجاء المحاولة لاحقاً.';
  }
}

$page_title = 'الإدارة العليا';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تسجيل الدخول - <?= e($page_title) ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --admin-primary: #1e3a8a;
            --admin-secondary: #3b82f6;
            --admin-accent: #f59e0b;
            --admin-dark: #0f172a;
            --light-bg: #f8fafc;
            --dark-text: #1e293b;
            --gray-border: #e2e8f0;
            --success: #10b981;
            --danger: #ef4444;
        }

        * {
            font-family: 'Cairo', sans-serif;
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
            direction: rtl;
        }

        .login-container {
            width: 100%;
            max-width: 1200px;
        }

        .login-card {
            background: white;
            border-radius: 24px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
            overflow: hidden;
            display: flex;
            min-height: 700px;
            border: 1px solid rgba(30, 58, 138, 0.1);
        }

        .login-left {
            flex: 1;
            background: linear-gradient(135deg, var(--admin-dark) 0%, var(--admin-primary) 100%);
            padding: 60px 40px;
            color: white;
            display: flex;
            flex-direction: column;
            justify-content: center;
            position: relative;
        }

        .login-left::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -20%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(59, 130, 246, 0.1) 0%, transparent 70%);
            opacity: 0.3;
        }

        .login-right {
            flex: 1.2;
            padding: 60px 50px;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }

        .welcome-text {
            font-size: 3rem;
            font-weight: 800;
            line-height: 1.2;
            margin-bottom: 20px;
            position: relative;
            z-index: 1;
            background: linear-gradient(135deg, #fff 0%, #bfdbfe 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .system-name {
            font-size: 1.6rem;
            font-weight: 600;
            opacity: 0.9;
            margin-bottom: 40px;
            position: relative;
            z-index: 1;
            color: #bfdbfe;
        }

        .admin-features-list {
            list-style: none;
            padding: 0;
            margin-top: 40px;
        }

        .admin-features-list li {
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            font-size: 1.1rem;
            position: relative;
            z-index: 1;
            background: rgba(255, 255, 255, 0.1);
            padding: 12px 20px;
            border-radius: 12px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .admin-features-list li i {
            background: linear-gradient(135deg, var(--admin-secondary), var(--admin-accent));
            width: 36px;
            height: 36px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-left: 15px;
            flex-shrink: 0;
        }

        .login-header {
            text-align: center;
            margin-bottom: 40px;
        }

        .login-icon {
            width: 90px;
            height: 90px;
            background: linear-gradient(135deg, var(--admin-primary), var(--admin-secondary));
            border-radius: 22px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            box-shadow: 0 15px 35px rgba(30, 58, 138, 0.3);
            position: relative;
        }

        .login-icon::after {
            content: '';
            position: absolute;
            top: -5px;
            right: -5px;
            bottom: -5px;
            left: -5px;
            border-radius: 26px;
            background: linear-gradient(135deg, var(--admin-primary), var(--admin-secondary));
            z-index: -1;
            opacity: 0.5;
            filter: blur(10px);
        }

        .login-icon i {
            font-size: 2.8rem;
            color: white;
        }

        .login-title {
            font-size: 2.5rem;
            font-weight: 800;
            background: linear-gradient(135deg, var(--admin-primary), var(--admin-secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 10px;
        }

        .login-subtitle {
            color: #475569;
            font-size: 1.2rem;
            margin-bottom: 30px;
            line-height: 1.6;
        }

        .admin-badge {
            display: inline-block;
            background: linear-gradient(135deg, var(--admin-accent), #fbbf24);
            color: #78350f;
            padding: 8px 20px;
            border-radius: 50px;
            font-weight: 700;
            font-size: 1rem;
            margin-bottom: 20px;
            letter-spacing: 1px;
            border: 2px solid rgba(245, 158, 11, 0.3);
        }

        .form-group {
            margin-bottom: 30px;
        }

        .input-container {
            position: relative;
            width: 100%;
        }

        .form-control {
            width: 100%;
            height: 60px;
            padding: 0 55px 0 15px;
            border: 2px solid var(--gray-border);
            border-radius: 14px;
            font-size: 1.1rem;
            transition: all 0.3s ease;
            background: white;
            color: var(--admin-dark);
            text-align: center;
            direction: rtl;
            font-weight: 500;
        }

        .form-control:focus {
            border-color: var(--admin-secondary);
            box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.15);
            outline: none;
            background: #f8fafc;
        }

        .input-container .input-icon {
            position: absolute;
            right: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--admin-secondary);
            font-size: 1.3rem;
            z-index: 2;
            width: 26px;
            height: 26px;
            display: flex;
            align-items: center;
            justify-content: center;
            pointer-events: none;
        }

        /* ===== زر إظهار/إخفاء كلمة المرور ===== */
        .password-toggle {
            position: absolute;
            left: 20px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #94a3b8;
            font-size: 1.2rem;
            padding: 12px;
            z-index: 3;
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            transition: all 0.3s ease;
            
            /* تعديلات المؤشر */
            cursor: pointer !important;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            outline: none;
        }

        .password-toggle:hover {
            color: var(--admin-secondary);
            background: rgba(59, 130, 246, 0.1);
            cursor: pointer !important;
        }

        .password-toggle i {
            font-size: 1.3rem;
            display: block;
            line-height: 1;
            pointer-events: none;
        }

        /* ===== زر الدخول ===== */
        .login-btn {
            width: 100%;
            height: 60px;
            background: linear-gradient(135deg, var(--admin-primary), var(--admin-secondary));
            color: white;
            border: none;
            border-radius: 14px;
            font-size: 1.2rem;
            font-weight: 700;
            transition: all 0.4s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 10px;
            letter-spacing: 0.5px;
            
            /* تعديلات المؤشر */
            cursor: pointer !important;
            position: relative;
            z-index: 1;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            user-select: none;
            -webkit-user-select: none;
            -moz-user-select: none;
            -ms-user-select: none;
            box-sizing: border-box;
            margin: 10px 0 0 0;
            padding: 0 20px;
            outline: none;
        }

        .login-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 15px 30px rgba(30, 58, 138, 0.25);
            background: linear-gradient(135deg, var(--admin-secondary), var(--admin-primary));
            cursor: pointer !important;
        }

        .login-btn:active {
            transform: translateY(-1px);
            cursor: pointer !important;
        }

        .login-btn i {
            margin-left: 12px;
            font-size: 1.3rem;
            pointer-events: none;
        }

        /* ===== زر العودة للرئيسية ===== */
        .back-home-link {
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, var(--admin-primary), var(--admin-secondary));
            color: white;
            text-decoration: none;
            font-weight: 700;
            margin-top: 25px;
            padding: 15px 30px;
            border-radius: 14px;
            transition: all 0.3s ease;
            text-align: center;
            width: 100%;
            max-width: 320px;
            margin-left: auto;
            margin-right: auto;
            box-shadow: 0 6px 12px rgba(30, 58, 138, 0.2);
            border: none;
            
            /* تعديلات المؤشر */
            cursor: pointer !important;
            position: relative;
            z-index: 1;
            box-sizing: border-box;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            outline: none;
        }

        .back-home-link:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(30, 58, 138, 0.3);
            text-decoration: none;
            color: white;
            background: linear-gradient(135deg, var(--admin-secondary), var(--admin-primary));
            cursor: pointer !important;
        }

        .back-home-link:active {
            transform: translateY(-1px);
            cursor: pointer !important;
        }

        .back-home-link i {
            margin-left: 12px;
            transition: transform 0.3s ease;
            pointer-events: none;
        }

        .back-home-link:hover i {
            transform: translateX(-5px);
        }

        /* ===== زر تبديل اللغة ===== */
        .lang-btn {
            background: rgba(255, 255, 255, 0.2);
            border: 2px solid rgba(255, 255, 255, 0.3);
            color: white;
            padding: 12px 24px;
            border-radius: 12px;
            font-weight: 700;
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
            
            /* تعديلات المؤشر */
            cursor: pointer !important;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            border: none;
            outline: none;
        }

        .lang-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateY(-2px);
            cursor: pointer !important;
        }

        .alert-custom {
            border-radius: 14px;
            padding: 18px 24px;
            font-size: 1.1rem;
            border: none;
            background: #fee2e2;
            color: var(--danger);
            margin-bottom: 30px;
            display: flex;
            align-items: center;
            text-align: right;
            border-right: 5px solid var(--danger);
        }

        .alert-custom.info {
            background: #dbeafe;
            color: #1d4ed8;
            border-right: 5px solid #1d4ed8;
        }

        .alert-custom i {
            margin-left: 15px;
            font-size: 1.4rem;
        }

        .footer-links {
            text-align: center;
            margin-top: 40px;
            color: #475569;
            font-size: 1rem;
            padding-top: 25px;
            border-top: 2px solid var(--gray-border);
        }

        .footer-links p {
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .footer-links p i {
            margin-left: 10px;
            color: var(--admin-secondary);
        }

        .language-switcher {
            position: absolute;
            top: 35px;
            left: 35px;
            z-index: 10;
        }

        .admin-warning {
            text-align: center;
            margin-top: 20px;
            padding: 12px;
            background: #fffbeb;
            border-radius: 10px;
            color: #92400e;
            font-size: 0.95rem;
            border: 1px solid #fbbf24;
        }

        .admin-warning i {
            color: #f59e0b;
            margin-left: 8px;
        }

        @media (max-width: 992px) {
            .login-card {
                flex-direction: column;
                min-height: auto;
            }
            
            .login-left, .login-right {
                padding: 40px 30px;
            }
            
            .welcome-text {
                font-size: 2.5rem;
            }
            
            .login-title {
                font-size: 2.2rem;
            }
            
            .back-home-link {
                max-width: 280px;
                padding: 14px 25px;
            }
        }

        @media (max-width: 576px) {
            .login-left, .login-right {
                padding: 30px 20px;
            }
            
            .login-title {
                font-size: 2rem;
            }
            
            .form-control {
                height: 55px;
                padding: 0 50px 0 15px;
                font-size: 1rem;
            }
            
            .login-btn {
                height: 55px;
                font-size: 1.1rem;
            }
            
            .welcome-text {
                font-size: 2.2rem;
            }
            
            .input-container .input-icon {
                right: 15px;
                font-size: 1.2rem;
            }
            
            .password-toggle {
                left: 15px;
                font-size: 1.1rem;
                width: 44px;
                height: 44px;
                padding: 10px;
            }
            
            .back-home-link {
                padding: 12px 20px;
                font-size: 1rem;
                max-width: 250px;
            }
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            
            <!-- Left Panel - Welcome Section -->
            <div class="login-left">
                <div class="language-switcher">
                    <button class="lang-btn" type="button">EN</button>
                </div>
                
                <h1 class="welcome-text">الإدارة العليا</h1>
                <p class="system-name">لوحة تحكم القيادات الإدارية</p>
                
                <ul class="admin-features-list">
                    <li>
                        <i class="fas fa-crown"></i>
                        <span>صلاحيات كاملة للنظام</span>
                    </li>
                    <li>
                        <i class="fas fa-chart-pie"></i>
                        <span>تقارير إحصائية شاملة</span>
                    </li>
                    <li>
                        <i class="fas fa-user-shield"></i>
                        <span>إدارة المستخدمين والصلاحيات</span>
                    </li>
                    <li>
                        <i class="fas fa-cogs"></i>
                        <span>إعدادات النظام المتقدمة</span>
                    </li>
                    <li>
                        <i class="fas fa-database"></i>
                        <span>نسخ احتياطي واستعادة البيانات</span>
                    </li>
                </ul>
                
                <div class="admin-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <span>هذه الصفحة مخصصة للإدارات العليا فقط</span>
                </div>
                
                <div style="margin-top: auto; opacity: 0.8; font-size: 0.9rem; color: #bfdbfe;">
                    <p>© 2026 نظام القبول الإلكتروني - قسم الإدارة العليا</p>
                </div>
            </div>
            
            <!-- Right Panel - Login Form -->
            <div class="login-right">
                <div class="login-header">
                    <div class="login-icon">
                        <i class="fas fa-user-cog"></i>
                    </div>
                    
                    <div class="admin-badge">
                        <i class="fas fa-shield-alt"></i> دخول الإدارة العليا
                    </div>
                    
                    <h2 class="login-title">تسجيل الدخول</h2>
                    <p class="login-subtitle">الرجاء إدخال بيانات الاعتماد الخاصة بك للوصول إلى لوحة التحكم الإدارية</p>
                </div>
                
                <?php if ($error): ?>
                    <div class="alert-custom" role="alert">
                        <i class="fas fa-exclamation-circle"></i>
                        <div><?= e($error) ?></div>
                    </div>
                <?php endif; ?>

                <?php if ($DEBUG): ?>
                    <div class="alert-custom info" role="alert">
                        <i class="fas fa-bug"></i>
                        <div>
                            <strong>وضع التصحيح نشط</strong>
                            <p class="mb-0">جميع الأخطاء معروضة على الشاشة.</p>
                        </div>
                    </div>
                <?php endif; ?>

                <form method="post" action="admin_login.php<?= $DEBUG ? '?debug=1' : '' ?>" autocomplete="off">
                    <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                    
                    <div class="form-group">
                        <div class="input-container">
                            <input type="text"
                                   class="form-control"
                                   id="username" 
                                   name="username"
                                   placeholder="اسم المستخدم للإدارة العليا"
                                   required 
                                   value="<?= e($username) ?>"
                                   autocomplete="off"
                                   maxlength="100">
                            <i class="input-icon fas fa-user-tie"></i>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <div class="input-container">
                            <input type="password"
                                   class="form-control"
                                   id="password" 
                                   name="password"
                                   placeholder="كلمة المرور الخاصة"
                                   required
                                   autocomplete="off"
                                   maxlength="255">
                            <button type="button" class="password-toggle" id="togglePassword">
                                <i class="fas fa-eye"></i>
                            </button>
                            <i class="input-icon fas fa-key"></i>
                        </div>
                    </div>
                    
                    <button class="login-btn" type="submit">
                        <i class="fas fa-sign-in-alt"></i>
                        دخول لوحة التحكم الإدارية
                    </button>
                </form>
                
                <div class="footer-links">
                    <p class="mb-2">
                        <i class="fas fa-info-circle"></i>
                        للمساعدة التقنية، الرجاء الاتصال بقسم الدعم الفني للإدارات العليا
                    </p>
                 
                    
                    <a href="https://uofg.edu.sd/Registration" class="back-home-link" rel="noopener noreferrer">
                        <i class="fas fa-arrow-right"></i>
                        العودة إلى الموقع الرئيسي
                    </a>
                    
                    <div style="margin-top: 20px; font-size: 0.9rem; color: #64748b;">
                        <i class="fas fa-exclamation-circle"></i>
                        <span>الوصول مقصور على المستخدمين ذوي الصلاحيات الإدارية العليا</span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Password visibility toggle
        document.getElementById('togglePassword').addEventListener('click', function(e) {
            e.preventDefault();
            const passwordInput = document.getElementById('password');
            const icon = this.querySelector('i');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });
        
        // Language switcher placeholder
        document.querySelector('.lang-btn').addEventListener('click', function() {
            alert('سيتم تفعيل ميزة الترجمة في تحديث لاحق');
        });
        
        // Form validation feedback
        document.querySelector('form').addEventListener('submit', function(e) {
            const username = document.getElementById('username').value.trim();
            const password = document.getElementById('password').value;
            
            if (!username || !password) {
                e.preventDefault();
                alert('يرجى ملء جميع الحقول المطلوبة');
                return false;
            }
            
            // منع الإرسال المتكرر للنموذج
            const submitBtn = this.querySelector('button[type="submit"]');
            if (submitBtn.disabled) {
                e.preventDefault();
                return false;
            }
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i> جاري تسجيل الدخول...';
        });
        
        // Auto-focus on username field
        document.addEventListener('DOMContentLoaded', function() {
            const usernameField = document.getElementById('username');
            if (usernameField && usernameField.value === '') {
                usernameField.focus();
            }
            
            // Add security warning for admin page
            console.log('%c ⚠️ تحذير أمني ⚠️', 'color: red; font-size: 18px; font-weight: bold;');
            console.log('%c هذه الصفحة مخصصة للإدارات العليا فقط. أي محاولة وصول غير مصرح بها ستسجل وتبلغ عنها.', 'color: #1e3a8a; font-size: 14px;');
            
            // تأكيد أن المؤشر يعمل على كل الأزرار
            const buttons = document.querySelectorAll('button, .back-home-link, .lang-btn, .password-toggle, .login-btn');
            buttons.forEach(btn => {
                btn.style.cursor = 'pointer';
            });
        });
        
        // منع النقر بالزر الأيمن (إجراء أمني إضافي)
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
        
        // منع حفظ كلمة المرور في المتصفح
        setTimeout(function() {
            document.getElementById('username').setAttribute('autocomplete', 'off');
            document.getElementById('password').setAttribute('autocomplete', 'off');
        }, 500);
    </script>
</body>
</html>