<?php
// admin_student_details.php - Individual Student Details
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

session_start();

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Check if student_id is provided
if (!isset($_GET['student_id'])) {
    header('Location: admin_students.php');
    exit;
}

$student_id = $_GET['student_id'];

try {
    require 'db.php';
    require 'helpers.php';

    $adminName = (string)($_SESSION['admin_username'] ?? 'الإدارة العليا');
    $mysqli1->set_charset('utf8mb4');
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

    // Get student details
    $student_sql = "
        SELECT 
            s.*,
            c.name as college_name,
            COALESCE(SUM(p.payment_amount), 0) as total_payments
        FROM students s
        LEFT JOIN colleges c ON s.college_id = c.id
        LEFT JOIN payments p ON s.student_id = p.student_id
        WHERE s.student_id = ?
        GROUP BY s.student_id
    ";
    
    $stmt = $mysqli1->prepare($student_sql);
    $stmt->bind_param('s', $student_id);
    $stmt->execute();
    $student = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$student) {
        throw new Exception('الطالب غير موجود');
    }

    // Get payment history
    $payments_sql = "
        SELECT * FROM payments 
        WHERE student_id = ? 
        ORDER BY created_at DESC
    ";
    $payments_stmt = $mysqli1->prepare($payments_sql);
    $payments_stmt->bind_param('s', $student_id);
    $payments_stmt->execute();
    $payments = $payments_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $payments_stmt->close();

    $data_loaded = true;

} catch (Exception $e) {
    $error_message = $e->getMessage();
    $data_loaded = false;
    $student = [];
    $payments = [];
}

include 'header.php';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تفاصيل الطالب - الإدارة العليا</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700;800&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --g-900: #1F3A2E;
            --g-700: #3E6B54;
            --purple-600: #8e44ad;
        }

        body {
            font-family: 'Cairo', system-ui, -apple-system, sans-serif;
            background: #f8f9fa;
            min-height: 100vh;
        }

        .dashboard-header {
            background: linear-gradient(135deg, var(--purple-600), #7d3c98);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }

        .info-card {
            background: white;
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
            border-right: 4px solid var(--purple-600);
            margin-bottom: 1.5rem;
        }

        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 0.75rem 0;
            border-bottom: 1px solid #e9ecef;
        }

        .info-row:last-child {
            border-bottom: none;
        }

        .info-label {
            font-weight: 600;
            color: var(--g-900);
        }

        .info-value {
            color: #6c757d;
        }

        .back-btn {
            background: var(--purple-600);
            color: white;
            border: none;
            padding: 0.75rem 1.5rem;
            border-radius: 0.75rem;
            font-weight: 700;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 0.5rem;
            font-weight: 600;
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="dashboard-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1 class="h3 mb-2">تفاصيل الطالب</h1>
                    <p class="mb-0">معلومات كاملة عن الطالب وسجلاته</p>
                </div>
                <div class="col-md-4 text-md-start">
                    <a href="admin_students.php" class="back-btn">
                        <i class="fas fa-arrow-right"></i> العودة للقائمة
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($error_message) ?></div>
        <?php endif; ?>

        <?php if ($data_loaded && $student): ?>
        <div class="row">
            <!-- Personal Information -->
            <div class="col-md-6">
                <div class="info-card">
                    <h4 class="mb-4">المعلومات الشخصية</h4>
                    
                    <div class="info-row">
                        <span class="info-label">الرقم الجامعي:</span>
                        <span class="info-value"><?= htmlspecialchars($student['student_id']) ?></span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">الاسم بالعربية:</span>
                        <span class="info-value"><?= htmlspecialchars($student['student_name'] ?? 'غير متوفر') ?></span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">الاسم بالإنجليزية:</span>
                        <span class="info-value"><?= htmlspecialchars($student['student_name_eng'] ?? 'غير متوفر') ?></span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">الكلية:</span>
                        <span class="info-value"><?= htmlspecialchars($student['college_name']) ?></span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">تاريخ التسجيل:</span>
                        <span class="info-value">
                            <?= !empty($student['created_at']) && $student['created_at'] != '0000-00-00 00:00:00' ? date('Y-m-d H:i', strtotime($student['created_at'])) : 'غير محدد' ?>
                        </span>
                    </div>
                </div>
            </div>

            <!-- Status Information -->
            <div class="col-md-6">
                <div class="info-card">
                    <h4 class="mb-4">حالة الطالب</h4>
                    
                    <div class="info-row">
                        <span class="info-label">الملف الشخصي:</span>
                        <span class="info-value">
                            <span class="status-badge <?= $student['profile_completed'] ? 'bg-success' : 'bg-danger' ?>">
                                <?= $student['profile_completed'] ? 'مكتمل' : 'غير مكتمل' ?>
                            </span>
                        </span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">الحالة الطبية:</span>
                        <span class="info-value">
                            <?php if ($student['medically_fit'] == 1): ?>
                                <span class="status-badge bg-success">لائق طبياً</span>
                            <?php elseif ($student['medically_fit'] == 0): ?>
                                <span class="status-badge bg-danger">غير لائق</span>
                            <?php else: ?>
                                <span class="status-badge bg-warning">قيد المراجعة</span>
                            <?php endif; ?>
                        </span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">حالة المقابلة:</span>
                        <span class="info-value">
                            <?php if ($student['interview_status'] == 'accepted'): ?>
                                <span class="status-badge bg-success">مقبول</span>
                            <?php elseif ($student['interview_status'] == 'rejected'): ?>
                                <span class="status-badge bg-danger">مرفوض</span>
                            <?php else: ?>
                                <span class="status-badge bg-warning">قيد الانتظار</span>
                            <?php endif; ?>
                        </span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">إجمالي المدفوعات:</span>
                        <span class="info-value text-success fw-bold">
                            <?= number_format($student['total_payments'], 2) ?>
                        </span>
                    </div>

                    <?php if (!empty($student['medical_notes'])): ?>
                    <div class="info-row">
                        <span class="info-label">ملاحظات طبية:</span>
                        <span class="info-value"><?= htmlspecialchars($student['medical_notes']) ?></span>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($student['interview_notes'])): ?>
                    <div class="info-row">
                        <span class="info-label">ملاحظات المقابلة:</span>
                        <span class="info-value"><?= htmlspecialchars($student['interview_notes']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Payment History -->
        <div class="info-card">
            <h4 class="mb-4">سجل المدفوعات</h4>
            
            <?php if (empty($payments)): ?>
                <p class="text-muted text-center py-3">لا توجد مدفوعات مسجلة لهذا الطالب</p>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="table-dark">
                            <tr>
                                <th>رقم المعاملة</th>
                                <th>المبلغ</th>
                                <th>تاريخ الدفع</th>
                                <th>العملة</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($payments as $payment): ?>
                                <tr>
                                    <td>#<?= htmlspecialchars($payment['id'] ?? '') ?></td>
                                    <td class="text-success fw-bold"><?= number_format($payment['payment_amount'], 2) ?></td>
                                    <td>
                                        <?= !empty($payment['created_at']) && $payment['created_at'] != '0000-00-00 00:00:00' ? date('Y-m-d H:i', strtotime($payment['created_at'])) : 'غير محدد' ?>
                                    </td>
                                    <td><?= htmlspecialchars($payment['payment_currency'] ?? 'SDG') ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
    </div>

    <?php include 'footer.php'; ?>
</body>
</html>