<?php
// batches.php - صفحة إدارة الدفعات (نسخة محسنة بالكامل)
require 'db.php';
require 'helpers.php';
require_committee();

// تفعيل وضع التصحيح (يمكن إزالته في الإنتاج)
if (isset($_GET['debug']) && $_GET['debug'] == 1 && ($_SESSION['user_role'] ?? '') === 'admin') {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    error_reporting(E_ALL & ~E_DEPRECATED & ~E_STRICT);
}

// تفعيل التقرير الصارم للأخطاء في MySQL
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

$committeeId   = (int)($_SESSION['committee_id'] ?? 0);
$committeeName = (string)($_SESSION['committee_name'] ?? '');
$canAll        = (bool)($_SESSION['committee_can_all'] ?? false);
$scopeCollege  = isset($_SESSION['committee_college_id']) ? (int)$_SESSION['committee_college_id'] : null;
$isAdmin       = ($_SESSION['user_role'] ?? '') === 'admin';

$mysqli1->set_charset('utf8mb4');

/* ===== دوال التحقق من صحة المدخلات ===== */
function validateInt($value, $default = null) {
    if ($value === null || $value === '') return $default;
    return filter_var($value, FILTER_VALIDATE_INT) !== false ? (int)$value : $default;
}

function validateString($value, $default = '', $maxLength = 191) {
    if ($value === null) return $default;
    $value = trim((string)$value);
    $value = strip_tags($value);
    $value = preg_replace('/[<>"\']/', '', $value);
    $value = substr($value, 0, $maxLength);
    return $value === '' ? $default : $value;
}


/* ===== دوال مساعدة ===== */
function getBatchStatistics() {
    global $mysqli1;
    
    try {
        $stats = [
            'total' => 0,
            'active' => 0,
            'inactive' => 0,
            'used_in_students' => 0
        ];
        
        // إجمالي الدفعات
        $result = $mysqli1->query("SELECT COUNT(*) as count FROM batches WHERE deleted_at IS NULL");
        $stats['total'] = $result->fetch_assoc()['count'];
        
        // الدفعات النشطة
        $result = $mysqli1->query("SELECT COUNT(*) as count FROM batches WHERE status = 1 AND deleted_at IS NULL");
        $stats['active'] = $result->fetch_assoc()['count'];
        
        // الدفعات غير النشطة
        $result = $mysqli1->query("SELECT COUNT(*) as count FROM batches WHERE status = 0 AND deleted_at IS NULL");
        $stats['inactive'] = $result->fetch_assoc()['count'];
        
        // الدفعات المستخدمة في جدول الطلاب
        $result = $mysqli1->query("
            SELECT COUNT(DISTINCT batch_id) as count 
            FROM students 
            WHERE batch_id IS NOT NULL AND batch_id IN (SELECT id FROM batches WHERE deleted_at IS NULL)
        ");
        $stats['used_in_students'] = $result->fetch_assoc()['count'];
        
        return $stats;
    } catch (Exception $e) {
        error_log("خطأ في جلب إحصائيات الدفعات: " . $e->getMessage());
        return null;
    }
}

/* ===== معالجة الإجراءات (إضافة، تعديل، حذف) ===== */

// إضافة دفعة جديدة
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    
    // التحقق من CSRF
    if (!verify_csrf($_POST['csrf'] ?? '')) {
        $_SESSION['error_message'] = 'رمز الجلسة غير صالح. الرجاء تحديث الصفحة والمحاولة مرة أخرى.';
        header('Location: batches.php');
        exit;
    }
    
    // معالجة الإضافة
    if ($_POST['action'] === 'add') {
        try {
            $value = validateString($_POST['value'] ?? '');
            $status = validateInt($_POST['status'] ?? 1, 1);
            
            // التحقق من صحة المدخلات
            $errors = [];
            
            if (empty($value)) {
                $errors[] = 'قيمة الدفعة مطلوبة';
            } elseif (strlen($value) > 191) {
                $errors[] = 'قيمة الدفعة طويلة جداً (الحد الأقصى 191 حرف)';
            } elseif (!preg_match('/^[a-zA-Z0-9\-\/\s]+$/', $value)) {
                $errors[] = 'قيمة الدفعة يجب أن تحتوي على أحرف وأرقام وشرطة فقط';
            }
            
            if (!empty($errors)) {
                $_SESSION['error_message'] = implode('<br>', $errors);
                header('Location: batches.php');
                exit;
            }
            
            // التحقق من عدم وجود الدفعة مسبقاً
            $check_stmt = $mysqli1->prepare("SELECT id FROM batches WHERE value = ? AND deleted_at IS NULL");
            if (!$check_stmt) {
                throw new Exception("فشل في تحضير استعلام التحقق: " . $mysqli1->error);
            }
            
            $check_stmt->bind_param("s", $value);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $_SESSION['error_message'] = 'هذه الدفعة موجودة مسبقاً';
            } else {
                $insert_stmt = $mysqli1->prepare("INSERT INTO batches (value, status, created_at, updated_at) VALUES (?, ?, NOW(), NOW())");
                if (!$insert_stmt) {
                    throw new Exception("فشل في تحضير استعلام الإضافة: " . $mysqli1->error);
                }
                
                $insert_stmt->bind_param("si", $value, $status);
                
                if ($insert_stmt->execute()) {
                    $_SESSION['success_message'] = 'تم إضافة الدفعة بنجاح';
                    
                    // تسجيل العملية
                    error_log("User {$_SESSION['user_id']} added new batch: $value");
                } else {
                    throw new Exception("فشل في تنفيذ الإضافة: " . $insert_stmt->error);
                }
                $insert_stmt->close();
            }
            $check_stmt->close();
            
        } catch (Exception $e) {
            error_log("خطأ في إضافة دفعة: " . $e->getMessage());
            
            if ($isAdmin) {
                $_SESSION['error_message'] = 'خطأ في النظام: ' . $e->getMessage();
            } else {
                $_SESSION['error_message'] = 'حدث خطأ في النظام. الرجاء المحاولة لاحقاً أو الاتصال بالدعم الفني.';
            }
        }
        
        header('Location: batches.php');
        exit;
    }
    
    // تحديث دفعة
    elseif ($_POST['action'] === 'edit') {
        try {
            $id = validateInt($_POST['id'] ?? 0);
            $value = validateString($_POST['value'] ?? '');
            $status = validateInt($_POST['status'] ?? 1, 1);
            
            // التحقق من صحة المدخلات
            $errors = [];
            
            if ($id <= 0) {
                $errors[] = 'معرف الدفعة غير صالح';
            }
            
            if (empty($value)) {
                $errors[] = 'قيمة الدفعة مطلوبة';
            } elseif (strlen($value) > 191) {
                $errors[] = 'قيمة الدفعة طويلة جداً (الحد الأقصى 191 حرف)';
            } elseif (!preg_match('/^[a-zA-Z0-9\-\/\s]+$/', $value)) {
                $errors[] = 'قيمة الدفعة يجب أن تحتوي على أحرف وأرقام وشرطة فقط';
            }
            
            if (!empty($errors)) {
                $_SESSION['error_message'] = implode('<br>', $errors);
                header('Location: batches.php');
                exit;
            }
            
            // التحقق من وجود الدفعة
            $check_exists = $mysqli1->prepare("SELECT id FROM batches WHERE id = ? AND deleted_at IS NULL");
            $check_exists->bind_param("i", $id);
            $check_exists->execute();
            if ($check_exists->get_result()->num_rows === 0) {
                $_SESSION['error_message'] = 'الدفعة غير موجودة';
                header('Location: batches.php');
                exit;
            }
            $check_exists->close();
            
            // التحقق من عدم وجود دفعة أخرى بنفس القيمة
            $check_stmt = $mysqli1->prepare("SELECT id FROM batches WHERE value = ? AND id != ? AND deleted_at IS NULL");
            $check_stmt->bind_param("si", $value, $id);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            
            if ($check_result->num_rows > 0) {
                $_SESSION['error_message'] = 'هذه الدفعة موجودة مسبقاً';
            } else {
                $update_stmt = $mysqli1->prepare("UPDATE batches SET value = ?, status = ?, updated_at = NOW() WHERE id = ?");
                $update_stmt->bind_param("sii", $value, $status, $id);
                
                if ($update_stmt->execute()) {
                    $_SESSION['success_message'] = 'تم تحديث الدفعة بنجاح';
                    error_log("User {$_SESSION['user_id']} updated batch ID: $id");
                } else {
                    throw new Exception("فشل في تحديث الدفعة: " . $update_stmt->error);
                }
                $update_stmt->close();
            }
            $check_stmt->close();
            
        } catch (Exception $e) {
            error_log("خطأ في تحديث دفعة: " . $e->getMessage());
            
            if ($isAdmin) {
                $_SESSION['error_message'] = 'خطأ في النظام: ' . $e->getMessage();
            } else {
                $_SESSION['error_message'] = 'حدث خطأ في النظام. الرجاء المحاولة لاحقاً.';
            }
        }
        
        header('Location: batches.php');
        exit;
    }
    
    // حذف دفعة (soft delete)
    elseif ($_POST['action'] === 'delete') {
        try {
            $id = validateInt($_POST['id'] ?? 0);
            
            if ($id <= 0) {
                $_SESSION['error_message'] = 'معرف الدفعة غير صالح';
                header('Location: batches.php');
                exit;
            }
            
            // التحقق من وجود الدفعة
            $check_exists = $mysqli1->prepare("SELECT id, value FROM batches WHERE id = ? AND deleted_at IS NULL");
            $check_exists->bind_param("i", $id);
            $check_exists->execute();
            $batch = $check_exists->get_result()->fetch_assoc();
            
            if (!$batch) {
                $_SESSION['error_message'] = 'الدفعة غير موجودة';
                header('Location: batches.php');
                exit;
            }
            $check_exists->close();
            
            // التحقق من عدم استخدام الدفعة في جدول students
            $check_stmt = $mysqli1->prepare("SELECT COUNT(*) as count FROM students WHERE batch_id = ? AND deleted_at IS NULL");
            $check_stmt->bind_param("i", $id);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            $count = $check_result->fetch_assoc()['count'];
            $check_stmt->close();
            
            if ($count > 0) {
                $_SESSION['error_message'] = "لا يمكن حذف هذه الدفعة لأنها مستخدمة في {$count} تسجيل طالب";
            } else {
                $delete_stmt = $mysqli1->prepare("UPDATE batches SET deleted_at = NOW() WHERE id = ?");
                $delete_stmt->bind_param("i", $id);
                
                if ($delete_stmt->execute()) {
                    $_SESSION['success_message'] = 'تم حذف الدفعة بنجاح';
                    error_log("User {$_SESSION['user_id']} deleted batch ID: $id, Value: {$batch['value']}");
                } else {
                    throw new Exception("فشل في حذف الدفعة: " . $delete_stmt->error);
                }
                $delete_stmt->close();
            }
            
        } catch (Exception $e) {
            error_log("خطأ في حذف دفعة: " . $e->getMessage());
            
            if ($isAdmin) {
                $_SESSION['error_message'] = 'خطأ في النظام: ' . $e->getMessage();
            } else {
                $_SESSION['error_message'] = 'حدث خطأ في النظام. الرجاء المحاولة لاحقاً.';
            }
        }
        
        header('Location: batches.php');
        exit;
    }
    
    // استعادة دفعة محذوفة (للمسؤول فقط)
    elseif ($_POST['action'] === 'restore' && $isAdmin) {
        try {
            $id = validateInt($_POST['id'] ?? 0);
            
            if ($id <= 0) {
                $_SESSION['error_message'] = 'معرف الدفعة غير صالح';
                header('Location: batches.php');
                exit;
            }
            
            $restore_stmt = $mysqli1->prepare("UPDATE batches SET deleted_at = NULL WHERE id = ?");
            $restore_stmt->bind_param("i", $id);
            
            if ($restore_stmt->execute() && $restore_stmt->affected_rows > 0) {
                $_SESSION['success_message'] = 'تم استعادة الدفعة بنجاح';
                error_log("Admin restored batch ID: $id");
            } else {
                $_SESSION['error_message'] = 'الدفعة غير موجودة أو لم يتم استعادتها';
            }
            $restore_stmt->close();
            
        } catch (Exception $e) {
            error_log("خطأ في استعادة دفعة: " . $e->getMessage());
            $_SESSION['error_message'] = 'حدث خطأ في استعادة الدفعة';
        }
        
        header('Location: batches.php');
        exit;
    }
}

/* ===== جلب جميع الدفعات ===== */
$search = validateString($_GET['search'] ?? '', '', 100);
$status_filter = isset($_GET['status']) && $_GET['status'] !== '' ? validateInt($_GET['status']) : null;
$show_deleted = $isAdmin && isset($_GET['show_deleted']) && $_GET['show_deleted'] == 1;

// جلب الإحصائيات
$statistics = getBatchStatistics();

$sql = "SELECT id, value, status, created_at, updated_at, deleted_at FROM batches WHERE 1=1";
$types = '';
$params = [];

if (!$show_deleted) {
    $sql .= " AND deleted_at IS NULL";
}

if (!empty($search)) {
    $sql .= " AND value LIKE ?";
    $types .= "s";
    $params[] = "%$search%";
}

if ($status_filter !== null) {
    $sql .= " AND status = ?";
    $types .= "i";
    $params[] = $status_filter;
}

$sql .= " ORDER BY 
    CASE 
        WHEN deleted_at IS NOT NULL THEN 1 
        ELSE 0 
    END,
    value DESC";

$stmt = $mysqli1->prepare($sql);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$batches = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// جلب الدفعات المحذوفة للإحصاء
if ($isAdmin) {
    $deleted_stmt = $mysqli1->prepare("SELECT COUNT(*) as count FROM batches WHERE deleted_at IS NOT NULL");
    $deleted_stmt->execute();
    $deleted_count = $deleted_stmt->get_result()->fetch_assoc()['count'];
    $deleted_stmt->close();
}

$page_title = 'إدارة الدفعات';
include 'header.php';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($page_title) ?> - <?= e($committeeName) ?></title>
    
    <!-- Bootstrap 5 RTL -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- DataTables -->
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.4.1/css/responsive.bootstrap5.min.css">
    
    <!-- SweetAlert2 -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    
    <!-- Google Fonts: Cairo -->
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    
    <style>
        :root {
            --primary-color: #1a5d1a;
            --secondary-color: #2e8b57;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
        }

        body {
            font-family: 'Cairo', sans-serif;
            background-color: #f3f4f6;
        }

        .content-wrapper {
            padding: 20px;
        }

        .page-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            padding: 25px;
            border-radius: 16px;
            margin-bottom: 25px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }

        .page-header h1 {
            margin: 0;
            font-size: 1.8rem;
            font-weight: 700;
        }

        .page-header p {
            margin: 10px 0 0;
            opacity: 0.9;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 20px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            border: 1px solid #e2e8f0;
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 24px rgba(0,0,0,0.1);
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 15px;
        }

        .stat-icon i {
            font-size: 24px;
        }

        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
            margin-bottom: 5px;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            line-height: 1.2;
        }

        .card-custom {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 25px;
        }

        .card-header-custom {
            background: transparent;
            border-bottom: 2px solid #f1f5f9;
            padding: 20px 25px;
            font-weight: 700;
            color: var(--primary-color);
            font-size: 1.2rem;
        }

        .card-header-custom i {
            margin-left: 10px;
            color: var(--secondary-color);
        }

        .btn-custom-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border: none;
            padding: 10px 25px;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-custom-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 15px rgba(26, 93, 26, 0.3);
            color: white;
        }

        .btn-custom-success {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-custom-warning {
            background: linear-gradient(135deg, #f59e0b, #d97706);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-custom-danger {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-custom-info {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-outline-custom {
            background: transparent;
            border: 2px solid var(--primary-color);
            color: var(--primary-color);
            padding: 8px 16px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .btn-outline-custom:hover {
            background: var(--primary-color);
            color: white;
        }

        .table th {
            background-color: #f8fafc;
            color: #475569;
            font-weight: 700;
            font-size: 0.95rem;
            border-bottom-width: 2px;
        }

        .table td {
            vertical-align: middle;
            color: #1e293b;
        }

        .status-badge {
            padding: 5px 12px;
            border-radius: 50px;
            font-size: 0.85rem;
            font-weight: 600;
            display: inline-block;
        }

        .status-active {
            background: linear-gradient(135deg, #d1fae5, #a7f3d0);
            color: #065f46;
        }

        .status-inactive {
            background: linear-gradient(135deg, #fee2e2, #fecaca);
            color: #991b1b;
        }

        .status-deleted {
            background: linear-gradient(135deg, #e2e8f0, #cbd5e1);
            color: #475569;
        }

        .action-buttons {
            display: flex;
            gap: 5px;
        }

        .action-buttons .btn {
            padding: 6px 12px;
            font-size: 0.85rem;
        }

        .filter-section {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #e2e8f0;
        }

        .modal-content {
            border-radius: 16px;
        }

        .modal-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border-radius: 16px 16px 0 0;
            padding: 20px 25px;
        }

        .modal-header .btn-close {
            filter: brightness(0) invert(1);
        }

        .modal-body {
            padding: 25px;
        }

        .modal-footer {
            border-top: 1px solid #e2e8f0;
            padding: 20px 25px;
        }

        .form-label {
            font-weight: 600;
            color: #475569;
            margin-bottom: 8px;
        }

        .form-control, .form-select {
            border-radius: 10px;
            border: 1px solid #e2e8f0;
            padding: 10px 15px;
            font-size: 0.95rem;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(26, 93, 26, 0.1);
        }

        .form-control.is-invalid {
            border-color: var(--danger-color);
            background-image: none;
        }

        .required::after {
            content: " *";
            color: var(--danger-color);
        }

        .alert-custom {
            border-radius: 12px;
            padding: 15px 20px;
            margin-bottom: 20px;
            border: none;
            display: flex;
            align-items: center;
        }

        .alert-custom i {
            margin-left: 10px;
            font-size: 1.2rem;
        }

        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border-right: 5px solid #059669;
        }

        .alert-danger {
            background: #fee2e2;
            color: #991b1b;
            border-right: 5px solid #dc2626;
        }

        .alert-warning {
            background: #fed7aa;
            color: #92400e;
            border-right: 5px solid #f59e0b;
        }

        .alert-info {
            background: #dbeafe;
            color: #1e40af;
            border-right: 5px solid #2563eb;
        }

        .badge-deleted {
            background: #e2e8f0;
            color: #475569;
            font-size: 0.75rem;
            padding: 3px 8px;
            border-radius: 50px;
            margin-right: 5px;
        }

        .deleted-row {
            background-color: #f8fafc;
            opacity: 0.8;
        }

        .deleted-row td {
            color: #94a3b8;
        }

        @media (max-width: 768px) {
            .page-header h1 {
                font-size: 1.5rem;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .table-responsive {
                border-radius: 12px;
            }
        }
    </style>
</head>
<body>
    <?php include 'navbar.php'; ?>
    
    <div class="content-wrapper" dir="rtl">
        <div class="container-fluid">
            
            <!-- Page Header -->
            <div class="page-header">
                <div class="d-flex align-items-center justify-content-between">
                    <div>
                        <h1><i class="fas fa-calendar-alt ms-2"></i> إدارة الدفعات</h1>
                        <p>إضافة وتعديل وحذف الدفعات الدراسية - <?= e($committeeName) ?></p>
                    </div>
                    <div>
                        <a href="admin_dashboard.php" class="btn btn-light">
                            <i class="fas fa-arrow-right ms-2"></i>
                            العودة للوحة التحكم
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Alert Messages -->
            <?php if (isset($_SESSION['success_message'])): ?>
                <div class="alert-custom alert-success">
                    <i class="fas fa-check-circle"></i>
                    <div><?= e($_SESSION['success_message']) ?></div>
                </div>
                <?php unset($_SESSION['success_message']); ?>
            <?php endif; ?>

            <?php if (isset($_SESSION['error_message'])): ?>
                <div class="alert-custom alert-danger">
                    <i class="fas fa-exclamation-circle"></i>
                    <div><?= $_SESSION['error_message'] ?></div>
                </div>
                <?php unset($_SESSION['error_message']); ?>
            <?php endif; ?>
            
            <!-- Statistics Cards -->
            <?php if ($statistics): ?>
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon" style="background: #e0f2fe; color: #0369a1;">
                        <i class="fas fa-calendar-alt"></i>
                    </div>
                    <div class="stat-label">إجمالي الدفعات</div>
                    <div class="stat-value"><?= $statistics['total'] ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: #dcfce7; color: #166534;">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-label">الدفعات النشطة</div>
                    <div class="stat-value"><?= $statistics['active'] ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: #fee2e2; color: #991b1b;">
                        <i class="fas fa-times-circle"></i>
                    </div>
                    <div class="stat-label">الدفعات غير النشطة</div>
                    <div class="stat-value"><?= $statistics['inactive'] ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: #fef9c3; color: #854d0e;">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-label">الدفعات التي يوجد بها طلاب</div>
                    <div class="stat-value"><?= $statistics['used_in_students'] ?></div>
                </div>
                
                <?php if ($isAdmin && isset($deleted_count)): ?>
                <div class="stat-card">
                    <div class="stat-icon" style="background: #e2e8f0; color: #475569;">
                        <i class="fas fa-trash-alt"></i>
                    </div>
                    <div class="stat-label">محذوفة</div>
                    <div class="stat-value"><?= $deleted_count ?></div>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <!-- Filter Section -->
            <div class="filter-section" dir="rtl">
                <form method="get" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">بحث</label>
                        <input type="text" class="form-control" name="search" placeholder="بحث بقيمة الدفعة..." 
                               value="<?= e($search) ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">الحالة</label>
                        <select class="form-select" name="status">
                            <option value="">جميع الحالات</option>
                            <option value="1" <?= $status_filter === 1 ? 'selected' : '' ?>>نشط</option>
                            <option value="0" <?= $status_filter === 0 ? 'selected' : '' ?>>غير نشط</option>
                        </select>
                    </div>
                    <?php if ($isAdmin): ?>
                    <div class="col-md-2">
                        <label class="form-label">عرض</label>
                        <div class="form-check mt-2">
                            <input class="form-check-input" type="checkbox" name="show_deleted" value="1" 
                                   id="show_deleted" <?= $show_deleted ? 'checked' : '' ?>>
                            <label class="form-check-label" for="show_deleted">
                                <i class="fas fa-trash-alt ms-1 text-danger"></i>
                                عرض المحذوفة
                            </label>
                        </div>
                    </div>
                    <?php endif; ?>
                    <div class="col-md-<?= $isAdmin ? '2' : '4' ?> d-flex align-items-end">
                        <button type="submit" class="btn btn-custom-primary w-100">
                            <i class="fas fa-filter ms-2"></i>
                            تصفية
                        </button>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <a href="batches.php" class="btn btn-outline-secondary w-100">
                            <i class="fas fa-sync-alt ms-2"></i>
                            إعادة تعيين
                        </a>
                    </div>
                </form>
            </div>
            
            <!-- Batches Card -->
            <div class="card card-custom">
                <div class="card-header-custom d-flex align-items-center justify-content-between">
                    <div>
                        <i class="fas fa-list"></i>
                        قائمة الدفعات
                        <?php if ($show_deleted): ?>
                            <span class="badge-deleted">
                                <i class="fas fa-trash-alt ms-1"></i>
                                معروضة المحذوفة
                            </span>
                        <?php endif; ?>
                    </div>
                    <div>
                        <button class="btn btn-custom-success" data-bs-toggle="modal" data-bs-target="#addBatchModal">
                            <i class="fas fa-plus-circle ms-2"></i>
                            إضافة دفعة جديدة
                        </button>
                    </div>
                </div>
                
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover" id="batchesTable">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>قيمة الدفعة</th>
                                    <th>الحالة</th>
                                    <th>تاريخ الإضافة</th>
                                    <th>آخر تحديث</th>
                                    <?php if ($show_deleted): ?>
                                    <th>تاريخ الحذف</th>
                                    <?php endif; ?>
                                    <th>الإجراءات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($batches)): ?>
                                    <tr>
                                        <td colspan="<?= $show_deleted ? '7' : '6' ?>" class="text-center text-muted py-4">
                                            <i class="fas fa-calendar-times fa-2x mb-3"></i>
                                            <p>لا توجد دفعات <?= $show_deleted ? 'محذوفة' : '' ?></p>
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($batches as $index => $batch): ?>
                                        <tr class="<?= $batch['deleted_at'] ? 'deleted-row' : '' ?>">
                                            <td class="fw-bold"><?= $index + 1 ?></td>
                                            <td>
                                                <?= e($batch['value']) ?>
                                                <?php if ($batch['deleted_at']): ?>
                                                    <span class="badge-deleted">
                                                        <i class="fas fa-trash-alt ms-1"></i>
                                                        محذوفة
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($batch['deleted_at']): ?>
                                                    <span class="status-badge status-deleted">
                                                        <i class="fas fa-trash ms-1"></i>
                                                        محذوفة
                                                    </span>
                                                <?php elseif ($batch['status'] == 1): ?>
                                                    <span class="status-badge status-active">
                                                        <i class="fas fa-check-circle ms-1"></i>
                                                        نشط
                                                    </span>
                                                <?php else: ?>
                                                    <span class="status-badge status-inactive">
                                                        <i class="fas fa-times-circle ms-1"></i>
                                                        غير نشط
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?= date('Y-m-d', strtotime($batch['created_at'])) ?></td>
                                            <td><?= $batch['updated_at'] ? date('Y-m-d', strtotime($batch['updated_at'])) : '-' ?></td>
                                            <?php if ($show_deleted && $batch['deleted_at']): ?>
                                            <td><?= date('Y-m-d', strtotime($batch['deleted_at'])) ?></td>
                                            <?php elseif ($show_deleted): ?>
                                            <td>-</td>
                                            <?php endif; ?>
                                            <td>
                                                <div class="action-buttons">
                                                    <?php if ($batch['deleted_at']): ?>
                                                        <?php if ($isAdmin): ?>
                                                        <form method="post" style="display: inline;" onsubmit="return confirmRestore()">
                                                            <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                                                            <input type="hidden" name="action" value="restore">
                                                            <input type="hidden" name="id" value="<?= (int)$batch['id'] ?>">
                                                            <button type="submit" class="btn btn-custom-success btn-sm" 
                                                                    data-bs-toggle="tooltip" title="استعادة">
                                                                <i class="fas fa-undo-alt"></i>
                                                            </button>
                                                        </form>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        <button class="btn btn-custom-info btn-sm" 
                                                                onclick="editBatch(<?= htmlspecialchars(json_encode($batch)) ?>)"
                                                                data-bs-toggle="tooltip" title="تعديل">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn btn-custom-danger btn-sm" 
                                                                onclick="deleteBatch(<?= (int)$batch['id'] ?>, '<?= e($batch['value']) ?>')"
                                                                data-bs-toggle="tooltip" title="حذف">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Batch Modal -->
    <div class="modal fade" id="addBatchModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog" dir="rtl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-plus-circle ms-2"></i>
                        إضافة دفعة جديدة
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" action="batches.php" id="addForm">
                    <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                    <input type="hidden" name="action" value="add">
                    
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label required">قيمة الدفعة</label>
                            <input type="text" class="form-control" name="value" id="add_value" required 
                                   placeholder="مثال: 2025-2026 أو 47" maxlength="191"
                                   pattern="[a-zA-Z0-9\-\/\s]+"
                                   title="يمكن استخدام الأحرف والأرقام والشرطة فقط">
                            <div class="form-text">أدخل قيمة الدفعة (يمكن استخدام الأحرف والأرقام والشرطة فقط)</div>
                            <div class="invalid-feedback">قيمة الدفعة مطلوبة وتقبل الأحرف والأرقام والشرطة فقط</div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">الحالة</label>
                            <select class="form-select" name="status">
                                <option value="1">نشط</option>
                                <option value="0">غير نشط</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times ms-2"></i>
                            إلغاء
                        </button>
                        <button type="submit" class="btn btn-custom-success">
                            <i class="fas fa-save ms-2"></i>
                            حفظ الدفعة
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Batch Modal -->
    <div class="modal fade" id="editBatchModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog" dir="rtl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit ms-2"></i>
                        تعديل الدفعة
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" action="batches.php" id="editForm">
                    <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit_id">
                    
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label required">قيمة الدفعة</label>
                            <input type="text" class="form-control" name="value" id="edit_value" required 
                                   maxlength="191" pattern="[a-zA-Z0-9\-\/\s]+"
                                   title="يمكن استخدام الأحرف والأرقام والشرطة فقط">
                            <div class="invalid-feedback">قيمة الدفعة مطلوبة وتقبل الأحرف والأرقام والشرطة فقط</div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">الحالة</label>
                            <select class="form-select" name="status" id="edit_status">
                                <option value="1">نشط</option>
                                <option value="0">غير نشط</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times ms-2"></i>
                            إلغاء
                        </button>
                        <button type="submit" class="btn btn-custom-primary">
                            <i class="fas fa-save ms-2"></i>
                            تحديث الدفعة
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Batch Modal -->
    <div class="modal fade" id="deleteBatchModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog" dir="rtl">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle ms-2"></i>
                        تأكيد الحذف
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" action="batches.php" id="deleteForm">
                    <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="delete_id">
                    
                    <div class="modal-body">
                        <p>هل أنت متأكد من حذف الدفعة: <strong id="delete_value"></strong>؟</p>
                        <p class="text-danger mb-0">
                            <i class="fas fa-info-circle ms-2"></i>
                            هذا الإجراء سينقل الدفعة إلى سلة المحذوفات ويمكن استعادتها لاحقاً.
                        </p>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times ms-2"></i>
                            إلغاء
                        </button>
                        <button type="submit" class="btn btn-danger">
                            <i class="fas fa-trash ms-2"></i>
                            تأكيد الحذف
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    
    <script>
        // Initialize DataTable
        $(document).ready(function() {
            $('#batchesTable').DataTable({
                language: {
                    url: '//cdn.datatables.net/plug-ins/1.13.4/i18n/ar.json'
                },
                order: [[0, 'desc']],
                pageLength: 25,
                columnDefs: [
                    { orderable: false, targets: -1 }
                ]
            });
            
            // Initialize tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function(tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });
        
        // Edit batch function
        function editBatch(batch) {
            document.getElementById('edit_id').value = batch.id;
            document.getElementById('edit_value').value = batch.value;
            document.getElementById('edit_status').value = batch.status;
            
            var editModal = new bootstrap.Modal(document.getElementById('editBatchModal'));
            editModal.show();
        }
        
        // Delete batch function
        function deleteBatch(id, value) {
            document.getElementById('delete_id').value = id;
            document.getElementById('delete_value').textContent = value;
            
            var deleteModal = new bootstrap.Modal(document.getElementById('deleteBatchModal'));
            deleteModal.show();
        }
        
        // Confirm restore function
        function confirmRestore() {
            return confirm('هل أنت متأكد من استعادة هذه الدفعة؟');
        }
        
        // Form validation
        function validateBatchValue(value) {
            const pattern = /^[a-zA-Z0-9\-\/\s]+$/;
            return pattern.test(value);
        }
        
        // Add form validation
        document.getElementById('addForm')?.addEventListener('submit', function(e) {
            const valueField = document.getElementById('add_value');
            const value = valueField.value.trim();
            
            if (!value) {
                e.preventDefault();
                valueField.classList.add('is-invalid');
                valueField.nextElementSibling?.classList.add('d-block');
                Swal.fire({
                    icon: 'error',
                    title: 'خطأ',
                    text: 'الرجاء إدخال قيمة الدفعة',
                    confirmButtonText: 'حسناً'
                });
            } else if (!validateBatchValue(value)) {
                e.preventDefault();
                valueField.classList.add('is-invalid');
                valueField.nextElementSibling?.classList.add('d-block');
                Swal.fire({
                    icon: 'error',
                    title: 'خطأ',
                    text: 'قيمة الدفعة يجب أن تحتوي على أحرف وأرقام وشرطة فقط',
                    confirmButtonText: 'حسناً'
                });
            }
        });
        
        // Edit form validation
        document.getElementById('editForm')?.addEventListener('submit', function(e) {
            const valueField = document.getElementById('edit_value');
            const value = valueField.value.trim();
            
            if (!value) {
                e.preventDefault();
                valueField.classList.add('is-invalid');
                Swal.fire({
                    icon: 'error',
                    title: 'خطأ',
                    text: 'الرجاء إدخال قيمة الدفعة',
                    confirmButtonText: 'حسناً'
                });
            } else if (!validateBatchValue(value)) {
                e.preventDefault();
                valueField.classList.add('is-invalid');
                Swal.fire({
                    icon: 'error',
                    title: 'خطأ',
                    text: 'قيمة الدفعة يجب أن تحتوي على أحرف وأرقام وشرطة فقط',
                    confirmButtonText: 'حسناً'
                });
            }
        });
        
        // Clear validation on input
        document.querySelectorAll('[id$="_value"]').forEach(field => {
            field.addEventListener('input', function() {
                this.classList.remove('is-invalid');
            });
        });
        
        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            document.querySelectorAll('.alert-custom').forEach(alert => {
                alert.style.transition = 'opacity 0.5s ease';
                alert.style.opacity = '0';
                setTimeout(() => alert.remove(), 500);
            });
        }, 5000);
        
        // Show success message with SweetAlert if exists
        <?php if (isset($_SESSION['success_message'])): ?>
        Swal.fire({
            icon: 'success',
            title: 'تم بنجاح',
            text: '<?= e($_SESSION['success_message']) ?>',
            timer: 3000,
            showConfirmButton: false
        });
        <?php endif; ?>
        
        // Show error message with SweetAlert if exists
        <?php if (isset($_SESSION['error_message'])): ?>
        Swal.fire({
            icon: 'error',
            title: 'خطأ',
            text: '<?= addslashes($_SESSION['error_message']) ?>',
            confirmButtonText: 'حسناً'
        });
        <?php endif; ?>
    </script>
    
    <?php include 'footer.php'; ?>
</body>
</html>