<?php
// medical_list.php — Doctor dashboard with filters/search/sort + flash
require 'db.php';
require 'helpers.php';
require_doctor();

$doctor = current_doctor();

/* ------- Flash ------- */
$flashOk  = flash_get('ok');
$flashErr = flash_get('err');

/* ------- Inputs ------- */
$status = $_GET['status'] ?? 'all';
$allowedStatus = ['all','waiting','approved','rejected'];
if (!in_array($status, $allowedStatus, true)) $status = 'all';

$college_id = $_GET['college_id'] ?? '';
$q = trim($_GET['q'] ?? '');
$sort = $_GET['sort'] ?? 'id_desc';
$allowedSort = ['id_desc','id_asc','medical_date_desc','medical_date_asc'];
if (!in_array($sort, $allowedSort, true)) $sort = 'id_desc';

$limit = 50;

/* ------- جلب قائمة الكليات ------- */
$colleges = [];
$colleges_stmt = $mysqli1->prepare("SELECT id, name FROM colleges ORDER BY name");
if ($colleges_stmt) {
    $colleges_stmt->execute();
    $colleges_result = $colleges_stmt->get_result();
    while ($college = $colleges_result->fetch_assoc()) {
        $colleges[$college['id']] = $college['name'];
    }
    $colleges_stmt->close();
}

/* ------- WHERE ------- */
$where = [];
$params = [];
$types  = '';

switch ($status) {
  case 'waiting': 
      $where[] = "s.medical_status = 'completed'";
      $where[] = "s.profile_completed = 1";
      $where[] = "s.medically_fit IS NULL";
      break;
  case 'approved':     
      $where[] = "s.medically_fit = 1";     
      $where[] = "s.medical_status = 'completed'";
      $where[] = "s.profile_completed = 1";
      break;
  case 'rejected':     
      $where[] = "s.medically_fit = 0";     
      $where[] = "s.medical_status = 'completed'";
      $where[] = "s.profile_completed = 1";
      break;
  case 'all':
  default: 
      break;
}

if (!empty($college_id) && is_numeric($college_id)) {
    $where[] = "s.college_id = ?";
    $types  .= "i";
    $params[] = $college_id;
}

if ($q !== '') {
    if (ctype_digit($q)) {
        $where[] = "s.student_id = ?";
        $types  .= "s";
        $params[] = $q;
    } else {
        $where[] = "(s.student_id LIKE ? OR s.student_name LIKE ? OR s.student_name_eng LIKE ?)";
        $types  .= "sss";
        $like = "%{$q}%";
        $params[] = $like; $params[] = $like; $params[] = $like;
    }
}

$whereSql = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

/* ------- ORDER ------- */
switch ($sort) {
  case 'id_asc':              $orderBy = "ORDER BY s.student_id ASC"; break;
  case 'medical_date_desc':   $orderBy = "ORDER BY m.medical_date DESC, s.student_id DESC"; break;
  case 'medical_date_asc':    $orderBy = "ORDER BY m.medical_date ASC, s.student_id ASC"; break;
  case 'id_desc':
  default:                    $orderBy = "ORDER BY s.student_id DESC"; break;
}

/* ------- Query ------- */
$sql = "
  SELECT 
    s.student_id, 
    s.student_name, 
    s.student_name_eng, 
    s.nationality,
    s.phone_number, 
    s.interview_status, 
    s.medically_fit,
    s.medical_checked_at,
    s.last_medical_exam,
    s.medical_status,
    s.profile_completed,
    s.college_id,
    c.name as college_name,
    m.medical_date,
    m.has_disability,
    m.has_chronic_diseases,
    m.has_psychological_diseases,
    m.has_vision_problems,
    m.has_hearing_problems,
    m.covid_vaccinated,
    m.has_diabetes,
    m.has_blood_pressure,
    m.has_family_diseases
  FROM students s
  LEFT JOIN student_medical_records m ON s.student_id = m.student_id
  LEFT JOIN colleges c ON s.college_id = c.id
  $whereSql
  $orderBy
  LIMIT $limit
";
$stmt = $mysqli1->prepare($sql);
if (!$stmt) { die("DB error."); }
if ($types !== '') { $stmt->bind_param($types, ...$params); }
$stmt->execute();
$res = $stmt->get_result();

/* ------- Counters for tabs ------- */
function count_by($mysqli1, $condSql = '1', $types = '', $params = []) {
    $sql = "SELECT COUNT(*) FROM students s WHERE ($condSql)";
    $st = $mysqli1->prepare($sql);
    if ($types) { $st->bind_param($types, ...$params); }
    $st->execute();
    $st->bind_result($c);
    $st->fetch();
    $st->close();
    return (int)$c;
}

$totalAll   = count_by($mysqli1, '1');
$totalWait  = count_by($mysqli1, "s.medical_status = 'completed' AND s.profile_completed = 1 AND s.medically_fit IS NULL");
$totalApproved = count_by($mysqli1, "s.medically_fit = 1 AND s.medical_status = 'completed' AND s.profile_completed = 1");
$totalRejected = count_by($mysqli1, "s.medically_fit = 0 AND s.medical_status = 'completed' AND s.profile_completed = 1");

if (!empty($college_id) && is_numeric($college_id)) {
    $totalAll = count_by($mysqli1, "s.college_id = ?", "i", [$college_id]);
    $totalWait = count_by($mysqli1, "s.medical_status = 'completed' AND s.profile_completed = 1 AND s.medically_fit IS NULL AND s.college_id = ?", "i", [$college_id]);
    $totalApproved = count_by($mysqli1, "s.medically_fit = 1 AND s.medical_status = 'completed' AND s.profile_completed = 1 AND s.college_id = ?", "i", [$college_id]);
    $totalRejected = count_by($mysqli1, "s.medically_fit = 0 AND s.medical_status = 'completed' AND s.profile_completed = 1 AND s.college_id = ?", "i", [$college_id]);
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<title>كشف الفحص الطبي - لوحة الطبيب</title>
<?php base_head_css(); ?>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
:root {
  --primary: #4361ee;
  --success: #06d6a0;
  --warning: #ffd166;
  --danger: #ef476f;
  --dark: #2b2d42;
  --light: #f8f9fa;
  --border: #e9ecef;
  --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
}

* {
  box-sizing: border-box;
}

body {
  background: #f5f7fa;
  min-height: 100vh;
  font-family: 'Cairo', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
  margin: 0;
  padding: 20px;
}
.filters-section,
.filter-label,
.form-control,
.filter-actions,
.btn {
  font-family: 'Cairo', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

.container {
  max-width: 1400px;
  margin: 0 auto;
}

/* الكارد الرئيسي */
.main-card {
  background: white;
  border-radius: 16px;
  box-shadow: var(--shadow);
  border: 1px solid var(--border);
  overflow: hidden;
}

/* الهيدر */
.dashboard-header {
  background: linear-gradient(135deg, #2b2d42, #4a4e69);
  color: white;
  padding: 30px;
  text-align: center;
  position: relative;
}

.header-content {
  position: relative;
  z-index: 2;
}

.welcome-text {
  font-size: 1.8rem;
  font-weight: 700;
  margin-bottom: 10px;
}

.subtitle {
  font-size: 1.1rem;
  opacity: 0.9;
}

.header-actions {
  position: absolute;
  left: 30px;
  top: 50%;
  transform: translateY(-50%);
  display: flex;
  gap: 10px;
}

/* الرسائل */
.flash-message {
  padding: 15px 20px;
  margin: 20px;
  border-radius: 12px;
  font-weight: 600;
  display: flex;
  align-items: center;
  gap: 10px;
  animation: slideIn 0.3s ease;
}

.flash-success {
  background: #d1fae5;
  color: #065f46;
  border: 1px solid #a7f3d0;
}

.flash-error {
  background: #fee2e2;
  color: #991b1b;
  border: 1px solid #fecaca;
}

@keyframes slideIn {
  from {
    opacity: 0;
    transform: translateY(-10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

/* الفلاتر */
.filters-section {
  padding: 25px;
  background: var(--light);
  border-bottom: 1px solid var(--border);
}

.filters-grid {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
  gap: 20px;
  margin-bottom: 20px;
}

.filter-group {
  display: flex;
  flex-direction: column;
  gap: 8px;
}

.filter-label {
  display: flex;
  align-items: center;
  gap: 8px;
  font-weight: 600;
  color: var(--dark);
  font-size: 0.9rem;
}

.filter-label i {
  color: var(--primary);
}

.form-control {
  width: 100%;
  padding: 12px 15px;
  border: 2px solid #e2e8f0;
  border-radius: 10px;
  font-size: 0.9rem;
  transition: all 0.3s ease;
  background: white;
}

.form-control:focus {
  outline: none;
  border-color: var(--primary);
  box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
}

.search-wrapper {
  position: relative;
}

.search-wrapper i {
  position: absolute;
  right: 15px;
  top: 50%;
  transform: translateY(-50%);
  color: #94a3b8;
}

.filter-actions {
  display: flex;
  gap: 10px;
  justify-content: flex-end;
}

.btn {
  padding: 10px 20px;
  border: none;
  border-radius: 10px;
  font-weight: 600;
  cursor: pointer;
  transition: all 0.3s ease;
  display: inline-flex;
  align-items: center;
  gap: 8px;
  text-decoration: none;
  font-size: 0.9rem;
}

.btn-primary {
  background: var(--primary);
  color: white;
}

.btn-primary:hover {
  background: #3a56d4;
  transform: translateY(-2px);
  box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3);
}

.btn-light {
  background: #64748b;
  color: white;
}

.btn-light:hover {
  background: #475569;
  transform: translateY(-2px);
}

/* التبويبات */
.tabs-section {
  padding: 20px 25px;
  background: white;
  border-bottom: 1px solid var(--border);
}

.tabs-container {
  display: flex;
  gap: 10px;
  flex-wrap: wrap;
}

.tab {
  padding: 12px 24px;
  border-radius: 12px;
  text-decoration: none;
  font-weight: 600;
  transition: all 0.3s ease;
  display: flex;
  align-items: center;
  gap: 8px;
  position: relative;
  overflow: hidden;
}

.tab::before {
  content: '';
  position: absolute;
  top: 0;
  left: 0;
  right: 0;
  height: 3px;
  background: var(--primary);
  transform: scaleX(0);
  transition: transform 0.3s ease;
}

.tab.active::before {
  transform: scaleX(1);
}

.tab.active {
  background: #f1f5f9;
  color: var(--primary);
}

.tab:not(.active) {
  background: white;
  color: #64748b;
  border: 1px solid var(--border);
}

.tab:not(.active):hover {
  background: #f8fafc;
  transform: translateY(-2px);
}

.count-badge {
  background: rgba(255, 255, 255, 0.9);
  padding: 4px 8px;
  border-radius: 20px;
  font-size: 0.8rem;
  font-weight: 700;
  min-width: 30px;
  text-align: center;
}

.tab.all .count-badge { background: #e2e8f0; color: #475569; }
.tab.waiting .count-badge { background: var(--warning); color: #92400e; }
.tab.approved .count-badge { background: var(--success); color: #065f46; }
.tab.rejected .count-badge { background: var(--danger); color: #991b1b; }

/* الجدول */
.table-section {
  padding: 0;
}

.data-table {
  width: 100%;
  border-collapse: collapse;
}

.data-table th {
  background: #f8fafc;
  padding: 15px 12px;
  text-align: right;
  font-weight: 700;
  color: var(--dark);
  border-bottom: 2px solid var(--border);
  font-size: 0.9rem;
}

.data-table td {
  padding: 15px 12px;
  border-bottom: 1px solid var(--border);
  transition: background 0.2s ease;
}

.data-table tr:hover td {
  background: #f8fafc;
}

.data-table tr:last-child td {
  border-bottom: none;
}

/* الباجات */
.status-badge {
  padding: 6px 12px;
  border-radius: 20px;
  font-size: 0.8rem;
  font-weight: 700;
  display: inline-flex;
  align-items: center;
  gap: 4px;
}

.badge-waiting {
  background: #fef3c7;
  color: #92400e;
}

.badge-approved {
  background: #d1fae5;
  color: #065f46;
}

.badge-rejected {
  background: #fee2e2;
  color: #991b1b;
}

.medical-tags {
  display: flex;
  gap: 4px;
  flex-wrap: wrap;
}

.medical-tag {
  padding: 3px 8px;
  border-radius: 6px;
  font-size: 0.7rem;
  font-weight: 600;
  background: #e2e8f0;
  color: #475569;
}

.medical-tag.yes {
  background: #dcfce7;
  color: #166534;
}

.action-btn {
  padding: 8px 16px;
  background: var(--primary);
  color: white;
  text-decoration: none;
  border-radius: 8px;
  font-size: 0.8rem;
  font-weight: 600;
  transition: all 0.3s ease;
  display: inline-flex;
  align-items: center;
  gap: 6px;
}

.action-btn:hover {
  background: #3a56d4;
  transform: translateY(-1px);
  box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3);
  color: white;
  text-decoration: none;
}

/* الفوتر */
.table-footer {
  padding: 20px 25px;
  background: #f8fafc;
  border-top: 1px solid var(--border);
  text-align: center;
  color: #64748b;
  font-size: 0.9rem;
}

/* Responsive */
@media (max-width: 768px) {
  body {
    padding: 10px;
  }
  
  .filters-grid {
    grid-template-columns: 1fr;
  }
  
  .dashboard-header {
    padding: 20px;
  }
  
  .header-actions {
    position: static;
    transform: none;
    justify-content: center;
    margin-top: 15px;
  }
  
  .tabs-container {
    justify-content: center;
  }
  
  .data-table {
    display: block;
    overflow-x: auto;
  }
}

/* تأثيرات إضافية */
.pulse {
  animation: pulse 2s infinite;
}

@keyframes pulse {
  0% { transform: scale(1); }
  50% { transform: scale(1.05); }
  100% { transform: scale(1); }
}

.fade-in {
  animation: fadeIn 0.5s ease;
}

@keyframes fadeIn {
  from { opacity: 0; }
  to { opacity: 1; }
}

/* تحسينات للشكل العام */
.student-name {
  font-weight: 600;
  color: var(--dark);
}

.student-id {
  font-family: 'Courier New', monospace;
  font-weight: 700;
  color: #374151;
}

.college-name {
  color: #6b7280;
  font-size: 0.9rem;
}

.phone-number {
  direction: ltr;
  text-align: left;
  font-family: 'Courier New', monospace;
}
</style>
</head>
<body>
<div class="container">
  <div class="main-card fade-in">
    <!-- الهيدر -->
    <div class="dashboard-header">
      <div class="header-content">
        <div class="welcome-text">
          <i class="fas fa-user-md pulse"></i>
          مرحباً د. <?php echo e($doctor['name']); ?>
        </div>
        <div class="subtitle">لوحة التحكم- إدارة الخدمات الصحية</div>
      </div>
<div class="header-actions">
  <a class="btn btn-light" href="medical_list.php?status=all">
    <i class="fas fa-sync-alt"></i>
    تحديث
  </a>
  <a class="btn btn-light" href="signout.php">
    <i class="fas fa-sign-out-alt"></i>
    خروج
  </a>
</div>
    </div>

    <!-- الرسائل -->
    <?php if ($flashOk): ?>
      <div class="flash-message flash-success">
        <i class="fas fa-check-circle"></i>
        <?php echo e($flashOk); ?>
      </div>
    <?php endif; ?>
    
    <?php if ($flashErr): ?>
      <div class="flash-message flash-error">
        <i class="fas fa-exclamation-triangle"></i>
        <?php echo e($flashErr); ?>
      </div>
    <?php endif; ?>

    <!-- الفلاتر -->
    <div class="filters-section">
      <form method="get" id="filtersForm">
        <input type="hidden" name="status" value="<?php echo e($status); ?>">
        
        <div class="filters-grid">
          <div class="filter-group">
            <div class="filter-label">
              <i class="fas fa-university"></i>
              <span>الكليّة</span>
            </div>
            <select name="college_id" class="form-control">
              <option value="">جميع الكليات</option>
              <?php foreach ($colleges as $id => $name): ?>
                <option value="<?php echo $id; ?>" <?php echo $college_id == $id ? 'selected' : ''; ?>>
                  <?php echo e($name); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="filter-group">
            <div class="filter-label">
              <i class="fas fa-search"></i>
              <span>البحث</span>
            </div>
            <div class="search-wrapper">
              <input type="text" name="q" value="<?php echo e($q); ?>" placeholder="ابحث برقم أو اسم الطالب..." class="form-control">
             </div>
          </div>

          <div class="filter-group">
            <div class="filter-label">
              <i class="fas fa-sort"></i>
              <span>الترتيب</span>
            </div>
            <select name="sort" class="form-control">
              <option value="id_desc" <?php echo $sort==='id_desc'?'selected':''; ?>>الأحدث أولاً</option>
              <option value="id_asc" <?php echo $sort==='id_asc'?'selected':''; ?>>الأقدم أولاً</option>
              <option value="medical_date_desc" <?php echo $sort==='medical_date_desc'?'selected':''; ?>>آخر فحص (حديث)</option>
              <option value="medical_date_asc" <?php echo $sort==='medical_date_asc'?'selected':''; ?>>آخر فحص (قديم)</option>
            </select>
          </div>
        </div>

        <div class="filter-actions">
          <button type="submit" class="btn btn-primary">
            <i class="fas fa-filter"></i>
            تطبيق الفلاتر
          </button>
          <a href="medical_list.php" class="btn btn-light">
            <i class="fas fa-times"></i>
            إعادة تعيين
          </a>
        </div>
      </form>
    </div>

    <!-- التبويبات -->
    <div class="tabs-section">
      <div class="tabs-container">
        <a class="tab all <?php echo $status==='all'?'active':''; ?>" 
           href="?status=all&sort=<?php echo e($sort); ?>&q=<?php echo urlencode($q); ?>&college_id=<?php echo urlencode($college_id); ?>">
          <i class="fas fa-users"></i>
          الكل
          <span class="count-badge"><?php echo $totalAll; ?></span>
        </a>
        
        <a class="tab waiting <?php echo $status==='waiting'?'active':''; ?>" 
           href="?status=waiting&sort=<?php echo e($sort); ?>&q=<?php echo urlencode($q); ?>&college_id=<?php echo urlencode($college_id); ?>">
          <i class="fas fa-clock"></i>
          في الانتظار
          <span class="count-badge"><?php echo $totalWait; ?></span>
        </a>
        
        <a class="tab approved <?php echo $status==='approved'?'active':''; ?>" 
           href="?status=approved&sort=<?php echo e($sort); ?>&q=<?php echo urlencode($q); ?>&college_id=<?php echo urlencode($college_id); ?>">
          <i class="fas fa-check-circle"></i>
          مقبول
          <span class="count-badge"><?php echo $totalApproved; ?></span>
        </a>
        
        <a class="tab rejected <?php echo $status==='rejected'?'active':''; ?>" 
           href="?status=rejected&sort=<?php echo e($sort); ?>&q=<?php echo urlencode($q); ?>&college_id=<?php echo urlencode($college_id); ?>">
          <i class="fas fa-times-circle"></i>
          مرفوض
          <span class="count-badge"><?php echo $totalRejected; ?></span>
        </a>
      </div>
    </div>

    <!-- الجدول -->
    <div class="table-section">
      <table class="data-table">
        <thead>
          <tr>
            <th>الرقم الجامعي</th>
            <th>اسم الطالب</th>
            <th>الكليّة</th>
            <th>الهاتف</th>
            <th>الحالة</th>
            <th>آخر فحص</th>
            <th>المعلومات الطبية</th>
            <th>الإجراء</th>
          </tr>
        </thead>
        <tbody>
        <?php while($row = $res->fetch_assoc()): ?>
          <?php
            $name  = $row['student_name'] ?: $row['student_name_eng'];
            $medically_fit = $row['medically_fit'];
            
            if ($medically_fit === null) { 
              $medTxt = 'في الانتظار'; 
              $medClass = 'badge-waiting';
              $medIcon = 'fas fa-clock';
            } elseif ((int)$medically_fit === 1) { 
              $medTxt = 'مقبول طبيًا';         
              $medClass = 'badge-approved';
              $medIcon = 'fas fa-check-circle';
            } else { 
              $medTxt = 'مرفوض طبيًا';     
              $medClass = 'badge-rejected';
              $medIcon = 'fas fa-times-circle';
            }
          ?>
          <tr class="fade-in">
            <td><span class="student-id"><?php echo e($row['student_id']); ?></span></td>
            <td><span class="student-name"><?php echo e($name); ?></span></td>
            <td><span class="college-name"><?php echo e($row['college_name'] ?? '—'); ?></span></td>
            <td><span class="phone-number"><?php echo e($row['phone_number']); ?></span></td>
            <td>
              <span class="status-badge <?php echo $medClass; ?>">
                <i class="<?php echo $medIcon; ?>"></i>
                <?php echo $medTxt; ?>
              </span>
            </td>
            <td><?php echo e($row['medical_checked_at'] ? date('Y-m-d H:i', strtotime($row['medical_checked_at'])) : '—'); ?></td>
            <td>
              <?php if ($row['medical_date']): ?>
                <div class="medical-tags">
                  <?php if ($row['has_disability'] === 'yes'): ?>
                    <span class="medical-tag yes" title="إعاقة حركية">إعاقة</span>
                  <?php endif; ?>
                  <?php if ($row['has_chronic_diseases'] === 'yes'): ?>
                    <span class="medical-tag yes" title="أمراض مزمنة">مزمن</span>
                  <?php endif; ?>
                  <?php if ($row['has_psychological_diseases'] === 'yes'): ?>
                    <span class="medical-tag yes" title="أمراض نفسية">نفسي</span>
                  <?php endif; ?>
                  <?php if ($row['has_vision_problems'] === 'yes'): ?>
                    <span class="medical-tag yes" title="مشاكل بصرية">بصر</span>
                  <?php endif; ?>
                  <?php if ($row['has_hearing_problems'] === 'yes'): ?>
                    <span class="medical-tag yes" title="مشاكل سمعية">سمع</span>
                  <?php endif; ?>
                </div>
              <?php else: ?>
                <span style="color: #94a3b8; font-size: 0.8rem;">—</span>
              <?php endif; ?>
            </td>
            <td>
              <a class="action-btn" href="medical_review.php?id=<?php echo urlencode($row['student_id']); ?>">
                <i class="fas fa-eye"></i>
                عرض
              </a>
            </td>
          </tr>
        <?php endwhile; ?>
        </tbody>
      </table>
    </div>

    <!-- الفوتر -->
    <div class="table-footer">
      <i class="fas fa-info-circle"></i>
      تم عرض أول <?php echo $limit; ?> طالب. استخدم البحث والتصفية للحصول على نتائج أكثر دقة.
    </div>
  </div>
</div>

<script>
<script>
// تأثيرات بسيطة للصفحة
document.addEventListener('DOMContentLoaded', function() {
  // تأثير للتبويبات
  const tabs = document.querySelectorAll('.tab');
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'));
      this.classList.add('active');
    });
  });
  
  // تأثير للزر عند الضغط
  const buttons = document.querySelectorAll('.btn, .action-btn');
  buttons.forEach(btn => {
    btn.addEventListener('mousedown', function() {
      this.style.transform = 'scale(0.95)';
    });
    btn.addEventListener('mouseup', function() {
      this.style.transform = '';
    });
  });

  // إصلاح زر التحديث
  const refreshBtn = document.querySelector('a[href="medical_list.php"]');
  if (refreshBtn) {
    refreshBtn.addEventListener('click', function(e) {
      e.preventDefault();
      window.location.href = 'medical_list.php?status=all';
    });
  }

  // إصلاح زر الخروج
  const logoutBtn = document.querySelector('a[href="signout.php?role=doctor"]');
  if (logoutBtn) {
    logoutBtn.addEventListener('click', function(e) {
      e.preventDefault();
      // تأكيد الخروج
      if (confirm('هل أنت متأكد من تسجيل الخروج؟')) {
        window.location.href = 'signout.php';
      }
    });
  }
});
</script>
<script>
// تأثيرات بسيطة للصفحة
document.addEventListener('DOMContentLoaded', function() {
  // تأثير للتبويبات
  const tabs = document.querySelectorAll('.tab');
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'));
      this.classList.add('active');
    });
  });
  
  // تأثير للزر عند الضغط
  const buttons = document.querySelectorAll('.btn, .action-btn');
  buttons.forEach(btn => {
    btn.addEventListener('mousedown', function() {
      this.style.transform = 'scale(0.95)';
    });
    btn.addEventListener('mouseup', function() {
      this.style.transform = '';
    });
  });
});
</script>
</body>
</html>