<?php
// /admission3333/reports/registered_students.php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../helpers.php';

// Set charset for mysqli connection to prevent SQL injection
$mysqli1->set_charset("utf8mb4");

// Check committee access
if (!isset($_SESSION['committee_id'])) {
    header('Location: ../login.php');
    exit;
}

require_committee();

$committeeId   = (int)($_SESSION['committee_id'] ?? 0);
$committeeName = (string)($_SESSION['committee_name'] ?? '');
$canAll        = (bool)($_SESSION['committee_can_all'] ?? false);
$scopeCollege  = isset($_SESSION['committee_college_id']) ? (int)$_SESSION['committee_college_id'] : null;

// Initialize variables
$students = [];
$error_message = '';
$success_message = '';

// Validate and sanitize input parameters
$filters_applied = isset($_GET['filter_applied']) && $_GET['filter_applied'] === '1';

// Get and validate filter parameters - with proper sanitization
$filter_college_id = isset($_GET['filter_college']) && ctype_digit($_GET['filter_college']) ? (int)$_GET['filter_college'] : null;
$filter_section_id = isset($_GET['filter_section']) && ctype_digit($_GET['filter_section']) ? (int)$_GET['filter_section'] : null;
$filter_program_id = isset($_GET['filter_program']) && ctype_digit($_GET['filter_program']) ? (int)$_GET['filter_program'] : null;
$filter_batch_id = isset($_GET['filter_batch']) && ctype_digit($_GET['filter_batch']) ? (int)$_GET['filter_batch'] : null;
$filter_semester_id = isset($_GET['filter_semester']) && ctype_digit($_GET['filter_semester']) ? (int)$_GET['filter_semester'] : null;

// Validate that semester is required when filters are applied
if ($filters_applied && empty($filter_semester_id)) {
    $error_message = "الرجاء اختيار فصل دراسي لعرض الطلاب المسجلين";
    $filters_applied = false;
}

// Apply scope restrictions
if (!$canAll && $scopeCollege !== null) {
    // Force college filter to user's scope
    $filter_college_id = $scopeCollege;
} elseif (!$canAll && $scopeCollege === null) {
    // No access
    $filter_college_id = -1; // This will return no results
}

// Get semester name for display
$semester_name = '';
$semester_code = '';
if ($filter_semester_id && $filter_semester_id > 0) {
    $semester_stmt = $mysqli1->prepare("SELECT name, code FROM semesters WHERE id = ? AND deleted_at IS NULL");
    if ($semester_stmt) {
        $semester_stmt->bind_param("i", $filter_semester_id);
        $semester_stmt->execute();
        $semester_result = $semester_stmt->get_result();
        if ($semester_data = $semester_result->fetch_assoc()) {
            $semester_name = $semester_data['name'] ?? '';
            $semester_code = $semester_data['code'] ?? '';
        }
        $semester_stmt->close();
    }
}

// Get filtered students ONLY if filters were applied and semester is selected
if ($filters_applied && !empty($filter_semester_id) && $filter_semester_id > 0) {
    
    // Build the students query with proper prepared statements - FIXED SQL INJECTION
    $sqlStudents = "
        SELECT DISTINCT
            s.student_id,
            s.student_name,
            s.student_name_eng,
            s.college_id,
            s.section_id,
            s.program_id,
            s.batch_id,
            c.name AS college_name,
            sec.name AS section_name,
            p.name AS program_name,
            b.value AS batch_name,
            COALESCE(reg_summary.total_paid_amount, 0) as total_paid_amount,
            reg_summary.last_registration_date,
            COALESCE(reg_summary.payment_status, 'غير مسجل') as payment_status,
            COALESCE(reg_summary.unpaid_count, 0) as unpaid_count,
            COALESCE(reg_summary.total_registrations, 0) as total_registrations
        FROM students s
        LEFT JOIN colleges c ON s.college_id = c.id AND c.deleted_at IS NULL
        LEFT JOIN sections sec ON s.section_id = sec.id AND sec.deleted_at IS NULL
        LEFT JOIN programs p ON s.program_id = p.id AND p.deleted_at IS NULL
        LEFT JOIN batches b ON s.batch_id = b.id AND b.deleted_at IS NULL
        LEFT JOIN (
            -- Get registration info for this semester
            SELECT 
                r.student_id,
                SUM(r.payment_amount) as total_paid_amount,
                MAX(r.created_at) as last_registration_date,
                COUNT(r.id) as total_registrations,
                SUM(CASE WHEN r.is_paid = 0 THEN 1 ELSE 0 END) as unpaid_count,
                CASE 
                    WHEN COUNT(r.id) = 0 THEN 'غير مسجل'
                    WHEN SUM(CASE WHEN r.is_paid = 0 THEN 1 ELSE 0 END) = 0 THEN 'مسجل كاملاً'
                    ELSE 'مسجل جزئياً'
                END as payment_status
            FROM registrations r
            WHERE r.semester_id = ? 
              AND r.deleted_at IS NULL
            GROUP BY r.student_id
        ) reg_summary ON reg_summary.student_id = s.student_id
        WHERE s.deleted_at IS NULL
    ";

    $students_params = [$filter_semester_id];
    $students_types = "i";

    // Apply scope restrictions
    if (!$canAll && $scopeCollege !== null && $scopeCollege > 0) {
        $sqlStudents .= " AND s.college_id = ?";
        $students_types .= "i";
        $students_params[] = $scopeCollege;
    }

    // Apply filters
    if ($filter_college_id && $filter_college_id > 0) {
        $sqlStudents .= " AND s.college_id = ?";
        $students_types .= "i";
        $students_params[] = $filter_college_id;
    }

    if ($filter_section_id && $filter_section_id > 0) {
        $sqlStudents .= " AND s.section_id = ?";
        $students_types .= "i";
        $students_params[] = $filter_section_id;
    }

    if ($filter_program_id && $filter_program_id > 0) {
        $sqlStudents .= " AND s.program_id = ?";
        $students_types .= "i";
        $students_params[] = $filter_program_id;
    }

    if ($filter_batch_id && $filter_batch_id > 0) {
        $sqlStudents .= " AND s.batch_id = ?";
        $students_types .= "i";
        $students_params[] = $filter_batch_id;
    }

    // Filter to include only students who have ANY registration in this semester
    $sqlStudents .= " AND EXISTS (
        SELECT 1 FROM registrations r2 
        WHERE r2.student_id = s.student_id 
        AND r2.semester_id = ? 
        AND r2.deleted_at IS NULL
    )";
    
    $students_types .= "i";
    $students_params[] = $filter_semester_id;

    $sqlStudents .= " ORDER BY 
        CASE WHEN reg_summary.payment_status = 'مسجل كاملاً' THEN 1
             WHEN reg_summary.payment_status = 'مسجل جزئياً' THEN 2
             WHEN reg_summary.payment_status = 'غير مسجل' THEN 3
             ELSE 4 END,
        reg_summary.last_registration_date DESC, 
        s.student_name
    LIMIT 1000";

    $students_stmt = $mysqli1->prepare($sqlStudents);
    if ($students_stmt) {
        $students_stmt->bind_param($students_types, ...$students_params);
        $students_stmt->execute();
        $students_result = $students_stmt->get_result();
        while ($row = $students_result->fetch_assoc()) {
            $students[] = $row;
        }
        $students_stmt->close();
    } else {
        $error_message = "حدث خطأ في جلب بيانات الطلاب";
    }
}

// Get filter options with prepared statements
$colleges = [];
$colleges_stmt = $mysqli1->prepare("SELECT id, name FROM colleges WHERE deleted_at IS NULL ORDER BY name");
if ($colleges_stmt) {
    $colleges_stmt->execute();
    $colleges_result = $colleges_stmt->get_result();
    while ($row = $colleges_result->fetch_assoc()) {
        $colleges[] = $row;
    }
    $colleges_stmt->close();
}

// Get sections for the selected college
$sections = [];
if ($filter_college_id && $filter_college_id > 0) {
    $sections_sql = "
        SELECT DISTINCT s.id, s.name, s.parent
        FROM sections s 
        WHERE s.parent = ? AND s.deleted_at IS NULL 
        ORDER BY s.name
    ";
    $sections_stmt = $mysqli1->prepare($sections_sql);
    if ($sections_stmt) {
        $sections_stmt->bind_param("i", $filter_college_id);
        $sections_stmt->execute();
        $sections_result = $sections_stmt->get_result();
        while ($row = $sections_result->fetch_assoc()) {
            $sections[] = $row;
        }
        $sections_stmt->close();
    }
}

// Get programs - FIXED SQL INJECTION VULNERABILITY
$programs = [];
$programs_query = "
    SELECT DISTINCT p.id, p.name 
    FROM programs p 
    WHERE p.deleted_at IS NULL
";
$programs_params = [];
$programs_types = "";

if ($filter_program_id && $filter_program_id > 0) {
    $programs_query .= " AND p.id = ?";
    $programs_types .= "i";
    $programs_params[] = $filter_program_id;
}

$programs_query .= " ORDER BY p.name";

$programs_stmt = $mysqli1->prepare($programs_query);
if ($programs_stmt) {
    if (!empty($programs_params)) {
        $programs_stmt->bind_param($programs_types, ...$programs_params);
    }
    $programs_stmt->execute();
    $programs_result = $programs_stmt->get_result();
    while ($row = $programs_result->fetch_assoc()) {
        $programs[] = $row;
    }
    $programs_stmt->close();
}

// Get batches
$batches = [];
$batches_stmt = $mysqli1->prepare("SELECT id, value FROM batches WHERE deleted_at IS NULL ORDER BY id DESC");
if ($batches_stmt) {
    $batches_stmt->execute();
    $batches_result = $batches_stmt->get_result();
    while ($row = $batches_result->fetch_assoc()) {
        $batches[] = $row;
    }
    $batches_stmt->close();
}

// Get semesters
$semesters = [];
$semesters_stmt = $mysqli1->prepare("SELECT id, name, code FROM semesters WHERE deleted_at IS NULL ORDER BY id DESC");
if ($semesters_stmt) {
    $semesters_stmt->execute();
    $semesters_result = $semesters_stmt->get_result();
    while ($row = $semesters_result->fetch_assoc()) {
        $semesters[] = $row;
    }
    $semesters_stmt->close();
}

// Get filter names for display
$college_name = "";
if ($filter_college_id && $filter_college_id > 0) {
    foreach ($colleges as $college) {
        if ($college['id'] == $filter_college_id) {
            $college_name = $college['name'];
            break;
        }
    }
}

$section_name = "";
if ($filter_section_id && $filter_section_id > 0 && !empty($sections)) {
    foreach ($sections as $section) {
        if ($section['id'] == $filter_section_id) {
            $section_name = $section['name'];
            break;
        }
    }
}

$program_name = "";
if ($filter_program_id && $filter_program_id > 0) {
    foreach ($programs as $program) {
        if ($program['id'] == $filter_program_id) {
            $program_name = $program['name'];
            break;
        }
    }
}

$batch_name = "";
if ($filter_batch_id && $filter_batch_id > 0) {
    foreach ($batches as $batch) {
        if ($batch['id'] == $filter_batch_id) {
            $batch_name = $batch['value'];
            break;
        }
    }
}

$display_semester_name = "";
if ($filter_semester_id && $filter_semester_id > 0 && $semester_name) {
    $display_semester_name = $semester_name . ' (' . $semester_code . ')';
}

// Calculate totals and statistics for registered students
$totalStudents = count($students);
$totalAmount = array_sum(array_column($students, 'total_paid_amount'));
$fullyRegistered = 0;
$partiallyRegistered = 0;
$notRegistered = 0;

foreach ($students as $student) {
    if ($student['payment_status'] === 'مسجل كاملاً') {
        $fullyRegistered++;
    } elseif ($student['payment_status'] === 'مسجل جزئياً') {
        $partiallyRegistered++;
    } elseif ($student['payment_status'] === 'غير مسجل') {
        $notRegistered++;
    }
}

// Include header with correct path
include __DIR__ . '/../header.php';
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>جميع الطلاب في الفصل الدراسي - تقرير</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body, h1,h2,h3,h4,h5,h6, p, a, button, label { 
            font-family: 'Cairo', system-ui, -apple-system, sans-serif !important; 
        }
        .card-arabic {
            border: 1px solid #e5e7eb;
            border-radius: 16px;
            background: #fff;
            box-shadow: 0 10px 24px rgba(0,0,0,.06);
            margin-bottom: 20px;
        }
        .filter-icon {
            width: 40px;
            height: 40px;
            background: #10b981;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.1rem;
        }
        .filter-status {
            font-size: 0.85rem;
            padding: 0.3rem 0.6rem;
            border-radius: 6px;
        }
        .filter-status.applied {
            background-color: #d1fae5;
            color: #065f46;
            border: 1px solid #a7f3d0;
        }
        .filter-status.not-applied {
            background-color: #fef3c7;
            color: #92400e;
            border: 1px solid #fde68a;
        }
        .info-message {
            background-color: #f0f9ff;
            border: 1px solid #bae6fd;
            color: #0369a1;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        .warning-message {
            background-color: #fee2e2;
            border: 1px solid #f87171;
            color: #991b1b;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        .student-table th {
            background-color: #f8f9fa;
            font-weight: 600;
            text-align: right;
        }
        .student-table td {
            text-align: right;
        }
        .badge-custom {
            padding: 5px 10px;
            border-radius: 20px;
            font-weight: 500;
        }
        .form-select, .form-control {
            border-radius: 8px;
            border: 1px solid #ced4da;
        }
        .form-select:focus, .form-control:focus {
            border-color: #10b981;
            box-shadow: 0 0 0 0.2rem rgba(16, 185, 129, 0.25);
        }
        .btn-primary {
            background: linear-gradient(135deg, #10b981, #059669);
            border: none;
            border-radius: 8px;
            padding: 8px 20px;
        }
        .btn-primary:hover {
            background: linear-gradient(135deg, #059669, #047857);
        }
        .table-responsive {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
        }
        .total-row {
            background-color: #f8f9fa;
            font-weight: bold;
        }
        .total-row td {
            border-top: 2px solid #dee2e6;
        }
        .chip {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        .chip.success { background: #d4edda; color: #155724; }
        .chip.danger { background: #f8d7da; color: #721c24; }
        .chip.warning { background: #fff3cd; color: #856404; }
        .chip.info { background: #d1ecf1; color: #0c5460; }
        .chip.secondary { background: #e9ecef; color: #495057; }
        .chip.light { background: #f8f9fa; color: #6c757d; }
        @media print {
            .no-print { display: none !important; }
            body { font-size: 12px; }
            table { font-size: 10px; }
        }
        .semester-warning {
            background-color: #f0fdf4;
            border-left: 4px solid #10b981;
            padding: 10px 15px;
            margin-bottom: 15px;
            border-radius: 6px;
        }
        .print-container {
            display: none;
        }
        .banner-img {
            max-width: 100%;
            height: auto;
            margin-bottom: 15px;
        }
        
        /* RTL Direction from right to left */
        body, .container, .card, .table, .form-control, .form-select {
            direction: rtl;
            text-align: right;
        }
        
        .ms-auto {
            margin-right: auto !important;
            margin-left: 0 !important;
        }
        
        .me-2 {
            margin-left: 0.5rem !important;
            margin-right: 0 !important;
        }
        
        .me-1 {
            margin-left: 0.25rem !important;
            margin-right: 0 !important;
        }
        
        .ms-2 {
            margin-right: 0.5rem !important;
            margin-left: 0 !important;
        }
        
        .ms-3 {
            margin-right: 1rem !important;
            margin-left: 0 !important;
        }
        
        .ms-auto {
            margin-right: auto !important;
            margin-left: 0 !important;
        }
        
        .text-end {
            text-align: left !important;
        }
        
        .text-right {
            text-align: right !important;
        }
        
        .border-end-0 {
            border-left: 0 !important;
            border-right: 1px solid #ced4da !important;
        }
        
        .border-start-0 {
            border-right: 0 !important;
            border-left: 1px solid #ced4da !important;
        }
        
        .input-group-text {
            border-radius: 0 8px 8px 0 !important;
        }
        
        .form-control {
            border-radius: 8px 0 0 8px !important;
        }
        
        .input-group {
            flex-direction: row-reverse;
        }
        
        .btn-group {
            flex-direction: row-reverse;
        }
        
        .d-flex {
            flex-direction: row-reverse;
        }
        
        .gap-2 > *:not(:last-child) {
            margin-left: 0.5rem;
            margin-right: 0;
        }
        
        .ps-3 {
            padding-right: 1rem !important;
            padding-left: 0 !important;
        }
        
        .py-4 {
            padding: 1.5rem !important;
        }
        
        .mb-4 {
            margin-bottom: 1.5rem !important;
        }
        
        .mb-3 {
            margin-bottom: 1rem !important;
        }
        
        .mb-2 {
            margin-bottom: 0.5rem !important;
        }
        
        .mb-1 {
            margin-bottom: 0.25rem !important;
        }
        
        .mt-3 {
            margin-top: 1rem !important;
        }
        
        .pe-3 {
            padding-left: 1rem !important;
            padding-right: 0 !important;
        }
        
        /* تصميم جديد لملخص التصفية */
        .filter-summary-card {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #dee2e6;
            border-radius: 16px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .filter-summary-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #10b981;
        }
        
        .filter-summary-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #10b981, #059669);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.3rem;
            margin-left: 15px;
        }
        
        .filter-summary-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: #1f2937;
            margin: 0;
        }
        
        .filter-summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .filter-item {
            background: white;
            padding: 12px;
            border-radius: 10px;
            border: 1px solid #e5e7eb;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .filter-item-label {
            font-size: 0.85rem;
            color: #6b7280;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .filter-item-value {
            font-size: 1rem;
            font-weight: 600;
            color: #1f2937;
            word-break: break-word;
        }
        
        .filter-conditions {
            background: white;
            padding: 15px;
            border-radius: 10px;
            border: 1px solid #e5e7eb;
            margin-top: 15px;
        }
        
        .filter-conditions-title {
            font-size: 0.9rem;
            font-weight: 600;
            color: #374151;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .filter-conditions-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .filter-conditions-list li {
            padding: 5px 0;
            font-size: 0.85rem;
            color: #4b5563;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .filter-conditions-list li:before {
            content: "✓";
            color: #10b981;
            font-weight: bold;
        }
        
        .no-filters-message {
            text-align: center;
            padding: 30px 20px;
            color: #6b7280;
        }
        
        .no-filters-icon {
            font-size: 3rem;
            color: #d1d5db;
            margin-bottom: 15px;
        }
        
        /* تصميم البطاقات الإحصائية */
        .stats-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            border: 1px solid #e5e7eb;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            transition: transform 0.2s ease;
        }
        
        .stats-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px -1px rgba(0, 0, 0, 0.15);
        }
        
        .stats-card .number {
            font-size: 2.2rem;
            font-weight: 800;
            margin-bottom: 8px;
            line-height: 1;
        }
        
        .stats-card .label {
            font-size: 0.9rem;
            color: #6c757d;
            margin-bottom: 0;
        }
        
        .stats-card-icon {
            width: 60px;
            height: 60px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 15px;
            font-size: 1.5rem;
        }
        
        .stats-card-icon.green {
            background: linear-gradient(135deg, #dcfce7, #bbf7d0);
            color: #22c55e;
        }
        
        .stats-card-icon.blue {
            background: linear-gradient(135deg, #dbeafe, #bfdbfe);
            color: #3b82f6;
        }
        
        .stats-card-icon.purple {
            background: linear-gradient(135deg, #f3e8ff, #e9d5ff);
            color: #a855f7;
        }
        
        .stats-card-icon.orange {
            background: linear-gradient(135deg, #ffedd5, #fed7aa);
            color: #f97316;
        }
        
        .stats-card-icon.red {
            background: linear-gradient(135deg, #fee2e2, #fecaca);
            color: #ef4444;
        }
        
        .export-options {
            background: white;
            border-radius: 12px;
            padding: 20px;
            border: 1px solid #e5e7eb;
            margin-bottom: 20px;
        }
        
        .export-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .export-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-left: 15px;
            font-size: 1.1rem;
        }
        
        .export-title {
            font-size: 1rem;
            font-weight: 600;
            color: #1f2937;
            margin: 0;
        }
        
        .export-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .btn-export {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 10px 20px;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s ease;
            border: none;
        }
        
        .btn-export-excel {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
        }
        
        .btn-export-excel:hover {
            background: linear-gradient(135deg, #059669, #047857);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }
        
        .btn-export-pdf {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            color: white;
        }
        
        .btn-export-pdf:hover {
            background: linear-gradient(135deg, #dc2626, #b91c1c);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
        }
        
        @media (max-width: 768px) {
            .filter-summary-grid {
                grid-template-columns: 1fr;
            }
            
            .export-buttons {
                flex-direction: column;
            }
            
            .btn-export {
                width: 100%;
            }
        }
        
        /* Print styles */
        @media print {
            body { 
                font-size: 12px !important; 
                direction: rtl;
                padding: 0;
                margin: 0;
            }
            
            .no-print { 
                display: none !important; 
            }
            
            .print-only { 
                display: block !important; 
            }
            
            .print-header {
                text-align: center;
                margin-bottom: 20px;
                padding-bottom: 15px;
            }
            
            .print-header h2 {
                font-size: 18px;
                margin: 10px 0;
                font-weight: bold;
            }
            
            .print-info {
                font-size: 11px;
                color: #666;
                margin-bottom: 10px;
            }
            
            .print-table {
                width: 100%;
                border-collapse: collapse;
                font-size: 10px;
                direction: rtl;
            }
            
            .print-table th {
                background-color: #f0f0f0 !important;
                border: 1px solid #000 !important;
                padding: 6px 4px !important;
                font-weight: bold !important;
                text-align: right !important;
            }
            
            .print-table td {
                border: 1px solid #000 !important;
                padding: 5px 4px !important;
                text-align: right !important;
            }
            
            .print-total {
                font-weight: bold !important;
                background-color: #f8f9fa !important;
                border-top: 2px solid #000 !important;
            }
            
            .print-footer {
                margin-top: 20px;
                padding-top: 10px;
                border-top: 1px solid #000;
                font-size: 10px;
                text-align: center;
                color: #666;
            }
            
            /* Hide all elements except print content */
            body > *:not(.print-container) {
                display: none !important;
            }
        }
        
        .payment-badge {
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .payment-fully {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .payment-partial {
            background-color: #fef3c7;
            color: #92400e;
        }
        
        .payment-none {
            background-color: #f3f4f6;
            color: #6b7280;
        }
        
        .status-legend {
            display: flex;
            gap: 15px;
            margin-top: 10px;
            flex-wrap: wrap;
        }
        
        .status-item {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 0.85rem;
        }
        
        .status-color {
            width: 12px;
            height: 12px;
            border-radius: 50%;
        }
        
        .status-color.fully { background-color: #10b981; }
        .status-color.partial { background-color: #f59e0b; }
        .status-color.none { background-color: #9ca3af; }
    </style>
</head>
<body>
    <!-- Hidden Print Container -->
    <div class="print-container" id="printContainer">
        <div class="print-header">
            <!-- Banner Image -->
            <img src="../img/banar3.jpg" alt="Banner" class="banner-img" onerror="this.style.display='none'">
            
            <h2>تقرير جميع الطلاب في الفصل الدراسي</h2>
            <div class="print-info">
                تاريخ التقرير: <?= date('Y-m-d H:i:s') ?><br>
                <?php if ($college_name): ?>الكلية: <?= htmlspecialchars($college_name, ENT_QUOTES, 'UTF-8') ?><br><?php endif; ?>
                <?php if ($program_name): ?>البرنامج: <?= htmlspecialchars($program_name, ENT_QUOTES, 'UTF-8') ?><br><?php endif; ?>
                <?php if ($display_semester_name): ?>الفصل الدراسي: <?= htmlspecialchars($display_semester_name, ENT_QUOTES, 'UTF-8') ?><br><?php endif; ?>
            </div>
        </div>
        
        <table class="print-table">
            <thead>
                <tr>
                    <th width="5%">#</th>
                    <th width="15%">رقم الطالب</th>
                    <th width="50%">اسم الطالب</th>
                    <th width="15%">الدفعة</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($students)): ?>
                    <tr>
                        <td colspan="5" class="text-center">
                            لا توجد بيانات للعرض
                        </td>
                    </tr>
                <?php else: ?>
                    <?php 
                    $printTotalAmount = 0;
                    $printTotalStudents = 0;
                    ?>
                    <?php foreach ($students as $index => $student): ?>
                        <?php
                        $printTotalAmount += $student['total_paid_amount'];
                        $printTotalStudents++;
                        ?>
                        <tr>
                            <td><?= $index + 1 ?></td>
                            <td><?= htmlspecialchars($student['student_id'], ENT_QUOTES, 'UTF-8') ?></td>
                            <td>
                                <?= htmlspecialchars($student['student_name'] ?: $student['student_name_eng'], ENT_QUOTES, 'UTF-8') ?>
                            </td>
                            <td><?= htmlspecialchars($student['batch_name'], ENT_QUOTES, 'UTF-8') ?></td>
                       
                        </tr>
                    <?php endforeach; ?>
                    
                  
                <?php endif; ?>
            </tbody>
        </table>
        
        <div class="print-footer">
            تم إنشاء التقرير تلقائياً من نظام إدارة القبول والتسجيل<br>
            الصفحة 1 من 1
        </div>
    </div>
    
    <div class="container py-4">
        <!-- Header -->
        <div class="d-flex align-items-center mb-4 no-print" dir="ltr">
            <h3 class="mb-0 fw-bold">تقرير جميع الطلاب في الفصل الدراسي</h3>
            <div class="ms-auto">
                <span class="filter-status <?= $filters_applied ? 'applied' : 'not-applied' ?> me-2">
                    <i class="fas fa-<?= $filters_applied ? 'check-circle' : 'exclamation-triangle' ?> me-1"></i>
                    <?= $filters_applied ? 'تم تطبيق التصفية' : 'لم يتم تطبيق أي تصفية' ?>
                </span>
                <a class="btn btn-outline-secondary" href="../student_affairs_dashboard.php">
                    <i class="fas fa-arrow-right me-2"></i>العودة للوحة التحكم
                </a>
            </div>
        </div>

        <?php if ($error_message): ?>
            <div class="warning-message no-print">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <?= htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8') ?>
            </div>
        <?php endif; ?>

        <?php if (!$filters_applied && !$error_message): ?>
            <div class="info-message no-print">
                <i class="fas fa-info-circle me-2"></i>
                <strong>تنبيه:</strong> لم يتم تطبيق أي تصفية. الرجاء اختيار عوامل التصفية والفصل الدراسي ثم النقر على "تطبيق التصفية" لعرض جميع الطلاب.
            </div>
        <?php endif; ?>

        <!-- Filter Card -->
        <div class="card card-arabic p-4 mb-4 no-print">
            <div class="row align-items-center">
                <div class="col-md-12">
                    <div class="d-flex align-items-center mb-4" dir="ltr">
                        <div class="filter-icon me-2" >
                            <i class="fas fa-filter"></i>
                        </div>
                        <h6 class="fw-bold mb-0 text-dark" >تصفية جميع الطلاب في الفصل الدراسي</h6>
                        <div class="ms-3">
                            <span class="badge bg-light text-dark border">
                                <i class="fas fa-sliders-h me-1"></i>
                                <?= $filters_applied ? 'تصفية مطبقة' : 'في انتظار التطبيق' ?>
                            </span>
                        </div>
                    </div>
                    
                    <form method="get" class="row g-3 align-items-end" id="filterForm">
                        <!-- Hidden field to indicate filters were applied -->
                        <input type="hidden" name="filter_applied" value="1">
                        
                        <!-- College Filter -->
                        <div class="col-lg-3 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-university me-1"></i>
                                    الكلية
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_college" id="filter_college" onchange="loadSections(this.value)">
                                        <option value="">اختر الكلية</option>
                                        <?php foreach ($colleges as $college): ?>
                                            <?php 
                                            // Don't show college filter if user has scope restriction
                                            if (!$canAll && $scopeCollege !== null && $college['id'] != $scopeCollege) {
                                                continue;
                                            }
                                            ?>
                                            <option value="<?= (int)$college['id'] ?>" <?= ($filter_college_id == $college['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($college['name'], ENT_QUOTES, 'UTF-8') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Section Filter (Dependent on College) -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-layer-group me-1"></i>
                                    القسم
                                </label>
                                <div class="input-group input-group-sm position-relative">
                                    <select class="form-control border-end-0" name="filter_section" id="filter_section" <?= !$filter_college_id ? 'disabled' : '' ?>>
                                        <option value="">جميع الأقسام</option>
                                        <?php if ($filter_college_id && !empty($sections)): ?>
                                            <?php foreach ($sections as $section): ?>
                                                <option value="<?= (int)$section['id'] ?>" <?= ($filter_section_id == $section['id']) ? 'selected' : '' ?>>
                                                    <?= htmlspecialchars($section['name'], ENT_QUOTES, 'UTF-8') ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-building"></i>
                                    </span>
                                </div>
                                <?php if (!$filter_college_id): ?>
                                    <small class="text-muted">الرجاء اختيار كلية أولاً</small>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Program Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-graduation-cap me-1"></i>
                                    البرنامج
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_program" id="filter_program">
                                        <option value="">جميع البرامج</option>
                                        <?php foreach ($programs as $program): ?>
                                            <option value="<?= (int)$program['id'] ?>" <?= ($filter_program_id == $program['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($program['name'], ENT_QUOTES, 'UTF-8') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-book"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Batch Filter -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-users me-1"></i>
                                    الدفعة
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_batch">
                                        <option value="">جميع الدفعات</option>
                                        <?php foreach ($batches as $batch): ?>
                                            <option value="<?= (int)$batch['id'] ?>" <?= ($filter_batch_id == $batch['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($batch['value'], ENT_QUOTES, 'UTF-8') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar-alt"></i>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Semester Filter (Required) -->
                        <div class="col-lg-2 col-md-6">
                            <div class="form-group">
                                <label class="form-label fw-semibold text-muted mb-2">
                                    <i class="fas fa-calendar-alt me-1"></i>
                                    الفصل الدراسي <span class="text-danger">*</span>  <small class="text-danger">مطلوب</small>
                                </label>
                                <div class="input-group input-group-sm">
                                    <select class="form-control border-end-0" name="filter_semester" required>
                                        <option value="">اختر الفصل الدراسي</option>
                                        <?php foreach ($semesters as $semester): ?>
                                            <option value="<?= (int)$semester['id'] ?>" <?= ($filter_semester_id == $semester['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($semester['name'] . ' (' . $semester['code'] . ')', ENT_QUOTES, 'UTF-8') ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <span class="input-group-text bg-light border-start-0">
                                        <i class="fas fa-calendar"></i>
                                    </span> 
                                </div>
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="col-12">
                            <div class="d-flex gap-2 pt-2">
                                <button type="submit" class="btn btn-primary btn-sm px-3">
                                    <i class="fas fa-filter me-1"></i>
                                    تطبيق التصفية
                                </button>
                                <a href="registered_students.php" class="btn btn-outline-secondary btn-sm px-3">
                                    <i class="fas fa-sync-alt me-1"></i>
                                    إعادة تعيين
                                </a>
                                <?php if ($filters_applied): ?>
                                    <span class="badge bg-success align-self-center px-3 py-2">
                                        <i class="fas fa-check me-1"></i>
                                        تم تطبيق التصفية
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Report Content -->
        <?php if ($filters_applied && !empty($filter_semester_id) && $filter_semester_id > 0): ?>
           
            <!-- Filter Summary Card -->
            <div class="filter-summary-card mb-4 no-print">
                <div class="filter-summary-header">
                    <div class="filter-summary-icon">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h5 class="filter-summary-title">ملخص التصفية المطبقة</h5>
                </div>
                
                <div class="filter-summary-grid">
                    <?php if ($filter_college_id && $filter_college_id > 0 && $college_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-university text-primary"></i>
                                الكلية
                            </div>
                            <div class="filter-item-value"><?= htmlspecialchars($college_name, ENT_QUOTES, 'UTF-8') ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_section_id && $filter_section_id > 0 && $section_name && $filter_college_id && !empty($sections)): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-layer-group text-success"></i>
                                القسم
                            </div>
                            <div class="filter-item-value"><?= htmlspecialchars($section_name, ENT_QUOTES, 'UTF-8') ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_program_id && $filter_program_id > 0 && $program_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-graduation-cap text-warning"></i>
                                البرنامج
                            </div>
                            <div class="filter-item-value"><?= htmlspecialchars($program_name, ENT_QUOTES, 'UTF-8') ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_batch_id && $filter_batch_id > 0 && $batch_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-users text-info"></i>
                                الدفعة
                            </div>
                            <div class="filter-item-value"><?= htmlspecialchars($batch_name, ENT_QUOTES, 'UTF-8') ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($filter_semester_id && $filter_semester_id > 0 && $display_semester_name): ?>
                        <div class="filter-item">
                            <div class="filter-item-label">
                                <i class="fas fa-calendar-alt text-danger"></i>
                                الفصل الدراسي
                            </div>
                            <div class="filter-item-value"><?= htmlspecialchars($display_semester_name, ENT_QUOTES, 'UTF-8') ?></div>
                        </div>
                    <?php endif; ?>
                    
                    <div class="filter-item">
                        <div class="filter-item-label">
                            <i class="fas fa-users text-success"></i>
                            عدد الطلاب
                        </div>
                        <div class="filter-item-value">
                            <span class="badge bg-success fs-6"><?= number_format($totalStudents) ?></span>
                        </div>
                    </div>
                </div>
                
                <div class="filter-conditions">
                    <div class="filter-conditions-title">
                        <i class="fas fa-info-circle text-info"></i>
                        معايير العرض الحالية:
                    </div>
                    <ul class="filter-conditions-list">
                        <li>جميع الطلاب المسجلين في الفصل الدراسي المحدد</li>
                        <?php if ($filter_college_id && $filter_college_id > 0): ?>
                            <li>الكلية: <?= htmlspecialchars($college_name, ENT_QUOTES, 'UTF-8') ?></li>
                        <?php endif; ?>
                        <?php if ($filter_section_id && $filter_section_id > 0): ?>
                            <li>القسم: <?= htmlspecialchars($section_name, ENT_QUOTES, 'UTF-8') ?></li>
                        <?php endif; ?>
                        <?php if ($filter_program_id && $filter_program_id > 0): ?>
                            <li>البرنامج: <?= htmlspecialchars($program_name, ENT_QUOTES, 'UTF-8') ?></li>
                        <?php endif; ?>
                        <?php if ($filter_batch_id && $filter_batch_id > 0): ?>
                            <li>الدفعة: <?= htmlspecialchars($batch_name, ENT_QUOTES, 'UTF-8') ?></li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>
            
            <!-- Export Options -->
            <div class="export-options mb-4 no-print">
                <div class="export-header">
                    <div class="export-icon">
                        <i class="fas fa-download"></i>
                    </div>
                    <h5 class="export-title">خيارات التصدير والطباعة</h5>
                </div>
                
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <div class="text-muted mb-3 mb-md-0">
                            <i class="fas fa-info-circle me-2"></i>
                            <span>تاريخ التقرير: <?= date('Y-m-d H:i:s') ?></span>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="export-buttons justify-content-md-end">
                            <button class="btn btn-export btn-export-excel" onclick="exportToExcel()">
                                <i class="fas fa-file-excel"></i>
                                تصدير لإكسل
                            </button>
                            <button class="btn btn-export btn-export-pdf" onclick="printReportPDF()">
                                <i class="fas fa-print"></i>
                                طباعة PDF
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Students Table -->
            <div class="card card-arabic p-4">
                <div class="d-flex align-items-center mb-3" dir="ltr">
                    <div class="filter-icon me-2">
                        <i class="fas fa-users"></i>
                    </div>
                    <h6 class="fw-bold mb-0">
                        قائمة جميع الطلاب في الفصل الدراسي
                        <?php if ($program_name): ?>
                            <span class="chip warning ms-2">
                                <i class="fas fa-graduation-cap me-1"></i>
                                <?= htmlspecialchars($program_name, ENT_QUOTES, 'UTF-8') ?>
                            </span>
                        <?php endif; ?>
                        <span class="chip success ms-2">
                            <i class="fas fa-calendar-alt me-1"></i>
                            <?= htmlspecialchars($display_semester_name, ENT_QUOTES, 'UTF-8') ?>
                        </span>
                    </h6>
                    <div class="ms-auto">
                        <span class="badge bg-success fs-6">
                            <i class="fas fa-users me-1"></i>
                            <?= number_format($totalStudents) ?> طالب
                        </span>
                    </div>
                </div>

                <div class="table-responsive" dir="rtl">
                    <table class="table table-striped student-table" id="studentsTable">
                        <thead>
                            <tr>
                                <th width="5%">#</th>
                                <th width="15%">رقم الطالب</th>
                                <th width="25%">اسم الطالب</th>
                                <th width="15%">الكلية</th>
                                <th width="15%">القسم</th>
                                <th width="15%">البرنامج</th>
                                <th width="10%">الدفعة</th>
                                <th width="15%">حالة التسجيل</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($students)): ?>
                                <tr>
                                    <td colspan="8" class="text-center text-muted py-4">
                                        <div class="py-5">
                                            <i class="fas fa-info-circle fa-3x text-muted mb-3"></i>
                                            <h5 class="text-muted">لا توجد بيانات</h5>
                                            <p class="text-muted">لا يوجد طلاب مسجلين في الفصل الدراسي: <?= htmlspecialchars($display_semester_name, ENT_QUOTES, 'UTF-8') ?></p>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php 
                                $tableAmount = 0;
                                ?>
                                <?php foreach ($students as $index => $student): ?>
                                    <?php
                                    $tableAmount += $student['total_paid_amount'];
                                    
                                    // Determine payment status badge class
                                    $paymentBadgeClass = '';
                                    if ($student['payment_status'] === 'مسجل كاملاً') {
                                        $paymentBadgeClass = 'payment-fully';
                                    } elseif ($student['payment_status'] === 'مسجل جزئياً') {
                                        $paymentBadgeClass = 'payment-partial';
                                    } else {
                                        $paymentBadgeClass = 'payment-none';
                                    }
                                    ?>
                                    <tr>
                                        <td class="fw-bold"><?= $index + 1 ?></td>
                                        <td><code><?= htmlspecialchars($student['student_id'], ENT_QUOTES, 'UTF-8') ?></code></td>
                                        <td style="text-align: right; direction: rtl;">
                                            <?= htmlspecialchars($student['student_name'] ?: $student['student_name_eng'], ENT_QUOTES, 'UTF-8') ?>
                                        </td>
                                        <td><?= htmlspecialchars($student['college_name'], ENT_QUOTES, 'UTF-8') ?></td>
                                        <td><?= htmlspecialchars($student['section_name'], ENT_QUOTES, 'UTF-8') ?></td>
                                        <td>
                                            <span class="chip info">
                                                <i class="fas fa-graduation-cap me-1"></i>
                                                <?= htmlspecialchars($student['program_name'], ENT_QUOTES, 'UTF-8') ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?= htmlspecialchars($student['batch_name'], ENT_QUOTES, 'UTF-8') ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="payment-badge <?= $paymentBadgeClass ?>">
                                                <?= htmlspecialchars($student['payment_status'], ENT_QUOTES, 'UTF-8') ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                
                                <!-- Status Legend -->
                                <tr class="no-print">
                                    <td colspan="8" class="pt-3">
                                        <div class="status-legend">
                                            <div class="status-item">
                                                <span class="status-color fully"></span>
                                                <span>مسجل كاملاً</span>
                                            </div>
                                            <div class="status-item">
                                                <span class="status-color partial"></span>
                                                <span>مسجل جزئياً</span>
                                            </div>
                                            <div class="status-item">
                                                <span class="status-color none"></span>
                                                <span>غير مسجل</span>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Simple section loading - reload page when college changes
        function loadSections(collegeId) {
            const sectionSelect = document.getElementById('filter_section');
            
            // Reset section select
            sectionSelect.innerHTML = '<option value="">جميع الأقسام</option>';
            
            if (!collegeId || collegeId === '') {
                sectionSelect.disabled = true;
                return;
            }
            
            // Create a form and submit it to reload the page
            const form = document.createElement('form');
            form.method = 'get';
            form.style.display = 'none';
            
            // Add college parameter
            const collegeInput = document.createElement('input');
            collegeInput.type = 'hidden';
            collegeInput.name = 'filter_college';
            collegeInput.value = collegeId;
            form.appendChild(collegeInput);
            
            // Add filter_applied parameter
            const appliedInput = document.createElement('input');
            appliedInput.type = 'hidden';
            appliedInput.name = 'filter_applied';
            appliedInput.value = '1';
            form.appendChild(appliedInput);
            
            // Add current section if exists
            const currentSection = "<?= $filter_section_id ?>";
            if (currentSection) {
                const sectionInput = document.createElement('input');
                sectionInput.type = 'hidden';
                sectionInput.name = 'filter_section';
                sectionInput.value = currentSection;
                form.appendChild(sectionInput);
            }
            
            // Add current program if exists
            const currentProgram = "<?= $filter_program_id ?>";
            if (currentProgram) {
                const programInput = document.createElement('input');
                programInput.type = 'hidden';
                programInput.name = 'filter_program';
                programInput.value = currentProgram;
                form.appendChild(programInput);
            }
            
            // Add current batch if exists
            const currentBatch = "<?= $filter_batch_id ?>";
            if (currentBatch) {
                const batchInput = document.createElement('input');
                batchInput.type = 'hidden';
                batchInput.name = 'filter_batch';
                batchInput.value = currentBatch;
                form.appendChild(batchInput);
            }
            
            // Add current semester if exists
            const currentSemester = "<?= $filter_semester_id ?>";
            if (currentSemester) {
                const semesterInput = document.createElement('input');
                semesterInput.type = 'hidden';
                semesterInput.name = 'filter_semester';
                semesterInput.value = currentSemester;
                form.appendChild(semesterInput);
            }
            
            document.body.appendChild(form);
            form.submit();
        }
        
        // Export to Excel function
        function exportToExcel() {
            if (!<?= $filters_applied ? 'true' : 'false' ?> || !<?= !empty($filter_semester_id) ? 'true' : 'false' ?>) {
                alert('الرجاء تطبيق التصفية أولاً مع اختيار فصل دراسي لعرض الطلاب');
                return;
            }
            
            // Get the table
            const table = document.getElementById('studentsTable');
            
            if (!table) {
                alert('لا توجد بيانات للتصدير');
                return;
            }
            
            // Clone the table to avoid modifying the original
            const exportTable = table.cloneNode(true);
            
            // Remove the status legend row if it exists
            const legendRow = exportTable.querySelector('tr.no-print');
            if (legendRow) {
                legendRow.remove();
            }
            
            // Create a new workbook
            const wb = XLSX.utils.book_new();
            
            // Create worksheet from table
            const ws = XLSX.utils.table_to_sheet(exportTable);
            
            // Set column widths
            const wscols = [
                {wch: 5},   // #
                {wch: 15},  // Student ID
                {wch: 25},  // Student Name
                {wch: 15},  // College
                {wch: 15},  // Section
                {wch: 15},  // Program
                {wch: 10},  // Batch
                {wch: 15},  // Payment Status
            ];
            ws['!cols'] = wscols;
            
            // Add worksheet to workbook
            XLSX.utils.book_append_sheet(wb, ws, "جميع الطلاب");
            
            // Generate filename
            let filename = "جميع_الطلاب_الفصل_الدراسي";
            <?php if ($program_name): ?>
                filename += "_<?= preg_replace('/[^\p{L}\p{N}\s]/u', '', $program_name) ?>";
            <?php endif; ?>
            <?php if ($display_semester_name): ?>
                filename += "_<?= preg_replace('/[^\p{L}\p{N}\s]/u', '', $display_semester_name) ?>";
            <?php endif; ?>
            filename += "_<?= date('Y-m-d') ?>.xlsx";
            
            // Clean filename (remove special characters)
            filename = filename.replace(/[<>:"/\\|?*]+/g, '_');
            
            // Save the file
            XLSX.writeFile(wb, filename);
        }
        
        // PDF Print function with banner
        function printReportPDF() {
            if (!<?= $filters_applied ? 'true' : 'false' ?> || !<?= !empty($filter_semester_id) ? 'true' : 'false' ?>) {
                alert('الرجاء تطبيق التصفية أولاً مع اختيار فصل دراسي لعرض الطلاب');
                return;
            }
            
            if (<?= count($students) ?> === 0) {
                alert('لا توجد بيانات للطباعة');
                return;
            }
            
            // Get the print container HTML
            const printContainer = document.getElementById('printContainer').outerHTML;
            
            // Create a new window for printing
            const printWindow = window.open('', '_blank', 'width=800,height=600');
            
            // Create the print content with banner
            const printContent = `
                <!DOCTYPE html>
                <html dir="rtl">
                <head>
                    <meta charset="UTF-8">
                    <title>تقرير جميع الطلاب في الفصل الدراسي</title>
                    <style>
                        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap');
                        
                        * {
                            font-family: 'Cairo', sans-serif !important;
                            box-sizing: border-box;
                        }
                        
                        body {
                            margin: 0;
                            padding: 15mm;
                            direction: rtl;
                            background: white;
                            font-size: 12px;
                        }
                        
                        .print-header {
                            text-align: center;
                            margin-bottom: 20px;
                            padding-bottom: 10px;
                        }
                        
                        .banner-img {
                            max-width: 100%;
                            height: auto;
                            margin-bottom: 15px;
                        }
                        
                        .print-header h2 {
                            font-size: 20px;
                            margin: 5px 0;
                            font-weight: bold;
                            color: #000;
                        }
                        
                        .print-info {
                            font-size: 11px;
                            color: #666;
                            margin-top: 10px;
                            line-height: 1.6;
                        }
                        
                        .print-table {
                            width: 100%;
                            border-collapse: collapse;
                            font-size: 10px;
                            margin-top: 15px;
                        }
                        
                        .print-table th {
                            background-color: #f0f0f0;
                            border: 1px solid #000;
                            padding: 8px 5px;
                            font-weight: bold;
                            text-align: right;
                            font-size: 11px;
                        }
                        
                        .print-table td {
                            border: 1px solid #000;
                            padding: 6px 5px;
                            text-align: right;
                        }
                        
                        .print-total {
                            font-weight: bold;
                            background-color: #f8f9fa;
                            border-top: 2px solid #000;
                        }
                        
                        .print-total td {
                            padding: 8px 5px;
                        }
                        
                        .print-footer {
                            margin-top: 20px;
                            padding-top: 10px;
                            border-top: 1px solid #666;
                            font-size: 10px;
                            text-align: center;
                            color: #666;
                            line-height: 1.6;
                        }
                        
                        @media print {
                            @page {
                                size: A4 landscape;
                                margin: 15mm;
                            }
                            
                            body { 
                                margin: 0;
                                padding: 0;
                            }
                            
                            .print-table {
                                font-size: 9px;
                            }
                            
                            .print-table th, 
                            .print-table td {
                                padding: 5px 3px;
                            }
                        }
                    </style>
                </head>
                <body>
                    ${printContainer}
                </body>
                </html>
            `;
            
            printWindow.document.write(printContent);
            printWindow.document.close();
            
            // Trigger print after content is loaded
            printWindow.onload = function() {
                printWindow.focus();
                setTimeout(function() {
                    printWindow.print();
                }, 500);
            };
        }
        
        // Initialize when page loads
        document.addEventListener('DOMContentLoaded', function() {
            const collegeSelect = document.getElementById('filter_college');
            const sectionSelect = document.getElementById('filter_section');
            
            // Enable/disable section select based on college selection
            if (collegeSelect && sectionSelect) {
                if (!collegeSelect.value) {
                    sectionSelect.disabled = true;
                } else {
                    sectionSelect.disabled = false;
                }
            }
        });
    </script>
    
    <!-- Load SheetJS library for Excel export -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
</body>
</html>